*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   FORTRAN routines for the spherical coordinate transformations used
*   by the FITS "World Coordinate System" (WCS) convention.
*
*   Summary of routines
*   -------------------
*   The spherical coordinate transformations are implemented via
*   separate subroutines for the transformation in each direction.
*
*   Forward transformation; SPHFWD
*   ------------------------------
*   Transform celestial coordinates to the native coordinates of a
*   projection.
*
*   Given:
*      LNG,LAT  D        Celestial longitude and latitude, in degrees.
*      EUL      D(6)     Euler angles for the transformation:
*                          1: Celestial longitude of the native pole, in
*                             degrees.
*                          2: Celestial colatitude of the native pole,
*                             or native colatitude of the celestial
*                             pole, in degrees.
*                          3: Native longitude of the celestial pole, in
*                             degrees.
*                          4: cos(EUL(2))
*                          5: sin(EUL(2))
*                          6: EUL(1) - EUL(3)
*
*   Returned:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*      IERR     I        Error status; 0 means success.
*
*   Reverse transformation; SPHREV
*   ------------------------------
*   Transform native coordinates of a projection to celestial
*   coordinates.
*
*   Given:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*      EUL      D(6)     Euler angles for the transformation:
*                          1: Celestial longitude of the native pole, in
*                             degrees.
*                          2: Celestial colatitude of the native pole,
*                             or native colatitude of the celestial
*                             pole, in degrees.
*                          3: Native longitude of the celestial pole, in
*                             degrees.
*                          4: cos(EUL(2))
*                          5: sin(EUL(2))
*                          6: EUL(3) - EUL(1)
*
*   Returned:
*      LNG,LAT  D        Celestial longitude and latitude, in degrees.
*      IERR     I        Error status; 0 means success.
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: sph.f,v 1.1 1995/01/31 03:20:39 mcalabre Exp $
*=======================================================================
      SUBROUTINE SPHFWD (LNG, LAT, EUL, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COSLAT, COSLNG, DLNG, EUL(6), LAT, LNG, PHI,
     *          SINLAT, SINLNG, THETA, X, Y

      DOUBLE PRECISION ASIND, ATAN2D, COSD, SIND
*-----------------------------------------------------------------------
      COSLAT = COSD(LAT)
      SINLAT = SIND(LAT)

      DLNG = LNG - EUL(1)
      COSLNG = COSD(DLNG)
      SINLNG = SIND(DLNG)

*     Compute native coordinates.
      X =  SINLAT*EUL(5) - COSLAT*EUL(4)*COSLNG
      Y = -COSLAT*SINLNG
      IF (X.NE.0D0 .OR. Y.NE.0D0) THEN
         PHI = EUL(3) + ATAN2D(Y, X)
      ELSE
*        Change of origin of longitude.
         PHI = EUL(6) + LNG
      END IF

*     Normalize.
      IF (PHI.GT.180D0) THEN
         PHI = PHI - 360D0
      ELSE IF (PHI.LT.-180D0) THEN
         PHI = PHI + 360D0
      END IF

      THETA = ASIND(SINLAT*EUL(4) + COSLAT*EUL(5)*COSLNG)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SPHREV (PHI, THETA, EUL, LNG, LAT, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COSPHI, COSTHE, DPHI, EUL(6), LAT, LNG, PHI,
     *          SINPHI, SINTHE, THETA, X, Y

      DOUBLE PRECISION ASIND, ATAN2D, COSD, SIND
*-----------------------------------------------------------------------
      COSTHE = COSD(THETA)
      SINTHE = SIND(THETA)

      DPHI = PHI - EUL(3)
      COSPHI = COSD(DPHI)
      SINPHI = SIND(DPHI)

*     Compute celestial coordinates.
      X =  SINTHE*EUL(5) - COSTHE*EUL(4)*COSPHI
      Y = -COSTHE*SINPHI
      IF (X.NE.0D0 .OR. Y.NE.0D0) THEN
         LNG = EUL(1) + ATAN2D(Y, X)
      ELSE
*        Change of origin of longitude.
         LNG = PHI - EUL(6)
      END IF

      IF (LNG.GT.360D0) THEN
         LNG = LNG - 360D0
      ELSE IF (LNG.LT.-360D0) THEN
         LNG = LNG + 360D0
      END IF

      LAT = ASIND(SINTHE*EUL(4) + COSTHE*EUL(5)*COSPHI)

      IERR = 0
      RETURN
      END
