*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   FORTRAN routines which implement the FITS World Coordinate System
*   (WCS) convention.
*
*   Summary of routines
*   -------------------
*   These routines are provided as drivers for the lower level spherical
*   coordinate transformation and projection routines.  There are
*   separate driver routines for the forward (WCSFWD) and reverse
*   (WCSREV) transformations.  A third routine (WCSSET) initializes
*   transformation parameters but need not be called explicitly - see
*   the explanation of WCS below.
*
*   Forward transformation; WCSFWD
*   ------------------------------
*   Compute (X,Y) coordinates in the plane of projection from celestial
*   coordinates (LNG,LAT).
*
*   Given:
*      PCODE    C*3      WCS projection code (see below)
*      LNG,LAT  D        Celestial longitude and latitude of the
*                        projected point, in degrees.
*
*   Given and returned:
*      WCS      D(10)    Spherical coordinate transformation parameters
*                        (see below)
*      PRJ      D(0:20)  Projection parameters (usage is described in
*                        the prologue to "proj.f").
*
*   Returned:
*      X,Y      D        Projected coordinates, "degrees".
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid coordinate transformation
*                              parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid value of (LNG,LAT).
*
*   Reverse transformation; WCSREV
*   ------------------------------
*   Compute the celestial coordinates (LNG,LAT) of the point with
*   projected coordinates (X,Y).
*
*   Given:
*      PCODE    C*3      WCS projection code.
*      X,Y      D        Projected coordinates, "degrees".
*
*   Given and returned:
*      WCS      D(10)    Coordinate transformation parameters (see
*                        below)
*      PRJ      D(0:20)  Projection parameters (usage is described in
*                        the prologue to "proj.f").
*
*   Returned:
*      LNG,LAT  D        Celestial longitude and latitude of the
*                        projected point, in degrees.
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid coordinate transformation
*                              parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid value of (LNG,LAT).
*
*   WCS projection codes
*   --------------------
*   Zenithals/azimuthals:
*      AZP: zenithal/azimuthal perspective
*      TAN: gnomonic
*      SIN: synthesis (generalized orthographic)
*      STG: stereographic
*      ARC: zenithal/azimuthal equidistant
*      ZPN: zenithal/azimuthal polynomial
*      ZEA: zenithal/azimuthal equal area
*      AIR: Airy
*
*   Cylindricals:
*      CYP: cylindrical perspective
*      CAR: Cartesian
*      MER: Mercator
*      CEA: cylindrical equal area
*
*   Conics:
*      COP: conic perspective
*      COD: conic equidistant
*      COE: conic equal area
*      COO: conic orthomorphic
*
*   Polyconics:
*      BON: Bonne
*      PCO: polyconic
*
*   Pseudo-cylindricals:
*      GLS: Sanson-Flamsteed (global sinusoidal)
*      PAR: parabolic
*      MOL: Mollweide
*
*   Conventional:
*      AIT: Hammer-Aitoff
*
*   Quad-cubes:
*      CSC: COBE quadrilateralized spherical cube
*      QSC: quadrilateralized spherical cube
*      TSC: tangential spherical cube
*
*   Coordinate transformation parameters; WCS
*   -----------------------------------------
*   The WCS array consists of the following:
*
*      WCS(1:3)
*         The first pair of values should be set to the celestial
*         longitude and latitude (usually right ascension and
*         declination) of the reference point of the projection.  The
*         third value is the native longitude of the pole of the
*         celestial coordinate system which corresponds to the FITS
*         keyword LONGPOLE.  Note that this has no default value and
*         should normally be set to 180 degrees.
*      WCS(4)
*         This serves as a flag which must be set to zero whenever any
*         of WCS(1:3) are set or changed.  This signals the
*         initialization routine (WCSSET) to recompute intermediaries.
*
*   The remaining elements of the WCS array are maintained by the
*   initialization routines and should not be modified.  This is done
*   for the sake of efficiency and to allow an arbitrary number of
*   contexts to be maintained simultaneously.
*
*      WCS(5:10)
*         Euler angles and associated intermediaries derived from the
*         coordinate reference values.
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: wcs.f,v 1.1 1995/01/31 03:20:39 mcalabre Exp $
*=======================================================================
      SUBROUTINE WCSSET (PCODE, WCS, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, IREF(25), NDX
      DOUBLE PRECISION CPHIP, WCS(10), SLAT0, X, Y
      CHARACTER PCODES*99, PCODE*3

      DOUBLE PRECISION ASIND, ATAN2D, COSD, SIND

      DATA PCODES(01:40) /'AZP TAN SIN STG ARC ZPN ZEA AIR CYP CAR '/
      DATA PCODES(41:80) /'MER CEA COP COD COE COO BON PCO GLS PAR '/
      DATA PCODES(81:99) /'AIT MOL CSC QSC TSC'/
      DATA IREF /0,0,0,0,0,0,0,0,1,1,1,1,0,0,0,0,1,1,1,1,1,1,1,1,1/
*-----------------------------------------------------------------------
*     Determine the projection type.
      NDX = (INDEX(PCODES,PCODE) - 1)/4 + 1
      IF (NDX.EQ.0) THEN
*        Unrecognized projection code.
         IERR = 1
         RETURN
      END IF

*     Compute celestial coordinates of the native pole.
      IF (IREF(NDX).EQ.0) THEN
*        Reference point is at the native pole.
         WCS(5) = WCS(1)
         WCS(6) = 90D0 - WCS(2)
      ELSE
*        Reference point is at the native origin.
         SLAT0 = SIND(WCS(2))
         CPHIP = COSD(WCS(3))
         IF (CPHIP.EQ.0D0) THEN
            IF (SLAT0.NE.0D0) THEN
               IERR = 1
               RETURN
            END IF
            WCS(6) = 90D0
         ELSE
            IF (ABS(SLAT0/CPHIP).GT.1D0) THEN
               IERR = 1
               RETURN
            END IF
            WCS(6) = ASIND(SLAT0/CPHIP)
            IF (WCS(6).LT.0D0) WCS(6) = 180D0 + WCS(6)
         END IF

         X = -CPHIP*COSD(WCS(6))
         Y =  SIND(WCS(3))
         IF (X.EQ.0D0 .AND. Y.EQ.0D0) THEN
            IF (MOD(WCS(3),360D0).EQ.0D0) THEN
               WCS(5) = WCS(1) - 90D0
            ELSE
               WCS(5) = WCS(1) + 90D0
            END IF
         ELSE
            WCS(5) = WCS(1) - ATAN2D(Y,X)
         END IF
      END IF
      WCS(7)  = WCS(3)
      WCS(8)  = COSD(WCS(6))
      WCS(9)  = SIND(WCS(6))
      WCS(10) = WCS(7) - WCS(5)
      WCS(4)  = DBLE(NDX)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE WCSFWD (PCODE, LNG, LAT, WCS, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION LAT, LNG, PHI, PRJ(0:20), WCS(10), THETA, X, Y
      CHARACTER PCODE*3
*-----------------------------------------------------------------------
      IF (WCS(4).EQ.0D0) THEN
         CALL WCSSET (PCODE, WCS, IERR)
         IF (IERR.NE.0) RETURN
      END IF

*     Compute native coordinates.
      CALL SPHFWD (LNG, LAT, WCS(5), PHI, THETA, IERR)

*     The computed GOTO is used instead of an IF-block for efficiency.
      GO TO ( 10,  20,  30,  40,  50,  60,  70,  80,
     *       110, 120, 130, 140,
     *       210, 220, 230, 240,
     *       310, 320, 330, 340, 350, 360, 370, 380, 390) NINT(WCS(4))

*     AZP: zenithal/azimuthal perspective.
 10   CALL AZPFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     TAN: gnomonic (tan).
 20   CALL TANFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     SIN: orthographic (sine).
 30   CALL SINFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     STG: stereographic.
 40   CALL STGFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     ARC: zenithal/azimuthal equidistant.
 50   CALL ARCFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     ZPN: zenithal/azimuthal polynomial.
 60   CALL ZPNFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     ZEA: zenithal/azimuthal equal area.
 70   CALL ZEAFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     AIR: Airy's zenithal projection.
 80   CALL AIRFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     CYP: cylindrical perspective.
 110  CALL CYPFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     CAR: Cartesian.
 120  CALL CARFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     MER: Mercator's.
 130  CALL MERFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     CEA: cylindrical equal area.
 140  CALL CEAFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     COP: conic perspective.
 210  CALL COPFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     COD: conic equidistant.
 220  CALL CODFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     COE: conic equal area.
 230  CALL COEFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     COO: conic orthomorphic.
 240  CALL COOFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     BON: Bonne's projection.
 310  CALL BONFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     PCO: polyconic.
 320  CALL PCOFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     GLS: Sanson-Flamsteed (global sinusoid).
 330  CALL GLSFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     PAR: parabolic.
 340  CALL PARFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     AIT: Hammer-Aitoff.
 350  CALL AITFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     MOL: Mollweide's projection.
 360  CALL MOLFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     CSC: COBE quadrilateralized spherical cube.
 370  CALL CSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     QSC: quadrilateralized spherical cube.
 380  CALL QSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

*     TSC: tangential spherical cube.
 390  CALL TSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      GO TO 400

 400  IF (IERR.EQ.1) THEN
         IERR = 2
      ELSE IF (IERR.EQ.2) THEN
         IERR = 3
      END IF

      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE WCSREV (PCODE, X, Y, WCS, PRJ, LNG, LAT, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION LAT, LNG, PHI, PRJ(0:20), THETA, WCS(10), X, Y
      CHARACTER PCODE*3
*-----------------------------------------------------------------------
      IF (WCS(4).EQ.0D0) THEN
         CALL WCSSET (PCODE, WCS, IERR)
         IF (IERR.NE.0) RETURN
      END IF

*     The computed GOTO is used instead of an IF-block for efficiency.
      GO TO ( 10,  20,  30,  40,  50,  60,  70,  80,
     *       110, 120, 130, 140,
     *       210, 220, 230, 240,
     *       310, 320, 330, 340, 350, 360, 370, 380, 390) NINT(WCS(4))

*     AZP: zenithal/azimuthal perspective.
 10   CALL AZPREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     TAN: gnomonic (tan).
 20   CALL TANREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     SIN: orthographic (sine).
 30   CALL SINREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     STG: stereographic.
 40   CALL STGREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     ARC: zenithal/azimuthal equidistant.
 50   CALL ARCREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     ZPN: zenithal/azimuthal polynomial.
 60   CALL ZPNREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     ZEA: zenithal/azimuthal equal area.
 70   CALL ZEAREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     AIR: Airy's zenithal projection.
 80   CALL AIRREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     CYP: cylindrical perspective.
 110  CALL CYPREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     CAR: Cartesian.
 120  CALL CARREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     MER: Mercator's.
 130  CALL MERREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     CEA: cylindrical equal area.
 140  CALL CEAREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     COP: conic perspective.
 210  CALL COPREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     COD: conic equidistant.
 220  CALL CODREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     COE: conic equal area.
 230  CALL COEREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     COO: conic orthomorphic.
 240  CALL COOREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     BON: Bonne's projection.
 310  CALL BONREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     PCO: polyconic.
 320  CALL PCOREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     GLS: Sanson-Flamsteed (global sinusoid).
 330  CALL GLSREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     PAR: parabolic.
 340  CALL PARREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     AIT: Hammer-Aitoff.
 350  CALL AITREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     MOL: Mollweide's projection.
 360  CALL MOLREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     CSC: COBE quadrilateralized spherical cube.
 370  CALL CSCREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     QSC: quadrilateralized spherical cube.
 380  CALL QSCREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     TSC: tangential spherical cube.
 390  CALL TSCREV (X, Y, PRJ, PHI, THETA, IERR)
      GO TO 400

*     Compute native coordinates.
 400  IF (IERR.EQ.1) THEN
         IERR = 2
      ELSE IF (IERR.EQ.2) THEN
         IERR = 3
      END IF
      IF (IERR.NE.0) RETURN

      CALL SPHREV (PHI, THETA, WCS(5), LNG, LAT, IERR)

      RETURN
      END
