*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995,1996 Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   FORTRAN routines which implement the FITS World Coordinate System
*   (WCS) convention.
*
*   Summary of routines
*   -------------------
*   WCSFWD and WCSREV are high level driver routines for the WCS linear
*   transformation, spherical coordinate transformation, and spherical
*   projection routines.
*
*   Given either the celestial longitude or latitude plus an element of
*   the pixel coordinate a hybrid routine, WCSMIX, iteratively solves
*   for the unknown elements.
*
*   An initialization routine, WCSSET, computes indices from the CTYPE
*   array but need not be called explicitly - see the explanation of
*   WCS(0) below.
*
*
*   Initialization routine; WCSSET
*   ------------------------------
*   Initializes elements of the WCS array which holds indices into the
*   coordinate arrays.  Note that this routine need not be called
*   directly; it will be invoked by WCSFWD and WCSREV if WCS(0) is zero.
*
*   Given:
*      NAXIS    I        Number of image axes.
*      CTYPE    C()*8    Coordinate axis types corresponding to the FITS
*                        CTYPEn header cards.
*
*   Returned:
*      WCS      I(0:3)   Indices for the celestial coordinates obtained
*                        by parsing the CTYPE() array (see below).
*      IERR     I        Error status
*                           0: Success.
*                           1: Inconsistent or unrecognized coordinate
*                              axis types.
*
*
*   Forward transformation; WCSFWD
*   ------------------------------
*   Compute the pixel coordinate for given world coordinates.
*
*   Given:
*      CTYPE    C()*8    Coordinate axis types corresponding to the FITS
*                        CTYPEn header cards.
*
*   Given or returned:
*      WCS      I(0:3)   Indices for the celestial coordinates obtained
*                        by parsing the CTYPE() array (see below).
*
*   Given:
*      WORLD    D()      World coordinates.  WORLD(WCS(1)) and
*                        WORLD(WCS(2)) are the celestial longitude and
*                        latitude, in degrees.
*
*   Given:
*      CRVAL    D()      Coordinate reference values corresponding to
*                        the FITS CRVALn header cards.
*
*   Given and returned:
*      CEL      D(10)    Spherical coordinate transformation parameters
*                        (usage is described in the prologue to
*                        "cel.f").
*
*   Returned:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*
*   Given and returned:
*      PRJ      D(0:20)  Projection parameters (usage is described in
*                        the prologue to "proj.f").
*
*   Returned:
*      IMGCRD   D()      Image coordinate.  IMGCRD(WCS(1)) and
*                        IMGCRD(WCS(2)) are the projected x-, and
*                        y-coordinates, in "degrees".  For quadcube
*                        projections with a CUBEFACE axis the face
*                        number is also returned in IMGCRD(WCS(3)).
*
*   Given and returned:
*      LIN      D()      Linear transformation parameters (usage is
*                        described in the prologue to "lin.f").
*
*   Returned:
*      PIXCRD   D()      Pixel coordinate.
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid coordinate transformation
*                              parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid world coordinate.
*                           4: Invalid linear transformation parameters.
*
*
*   Reverse transformation; WCSREV
*   ------------------------------
*   Compute world coordinates for a given pixel coordinate.
*
*   Given:
*      CTYPE    C()*8    Coordinate axis types corresponding to the FITS
*                        CTYPEn header cards.
*
*   Given or returned:
*      WCS      I(0:3)   Indices for the celestial coordinates obtained
*                        by parsing the CTYPE() array (see below).
*
*   Given:
*      PIXCRD   D()      Pixel coordinate.
*
*   Given and returned:
*      LIN      D()      Linear transformation parameters (usage is
*                        described in the prologue to "lin.f").
*
*   Returned:
*      IMGCRD   D()      Image coordinate.  IMGCRD(WCS(1)) and
*                        IMGCRD(WCS(2)) are the projected x-, and
*                        y-coordinates, in "degrees".
*
*   Given and returned:
*      PRJ      D(0:20)  Projection parameters (usage is described in
*                        the prologue to "proj.f").
*
*   Returned:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*
*   Given:
*      CRVAL    D()      Coordinate reference values corresponding to
*                        the FITS CRVALn header cards.
*
*   Given and returned:
*      CEL      D(10)    Spherical coordinate transformation parameters
*                        (usage is described in the prologue to
*                        "cel.f").
*
*   Returned:
*      WORLD    D()      World coordinates.  WORLD(WCS(1)) and
*                        WORLD(WCS(2)) are the celestial longitude and
*                        latitude, in degrees.
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid coordinate transformation
*                              parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid pixel coordinate.
*                           4: Invalid linear transformation parameters.
*
*
*   Hybrid transformation; WCSMIX
*   -----------------------------
*   Given either the celestial longitude or latitude plus an element of
*   the pixel coordinate solve for the remaining elements by iterating
*   on the unknown celestial coordinate element using WCSFWD.
*
*   Given:
*      CTYPE    C()*8    Coordinate axis types corresponding to the FITS
*                        CTYPEn header cards.
*
*   Given or returned:
*      WCS      I(0:3)   Indices for the celestial coordinates obtained
*                        by parsing the CTYPE() array (see below).
*
*   Given:
*      MIXPIX   I        Which element of the pixel coordinate is given.
*      MIXCEL   I        Which element of the celestial coordinate is
*                        given:
*                           1: Celestial longitude is given in
*                              WORLD(WCS(1)), latitude returned in
*                              WORLD(WCS(2)).
*                           2: Celestial latitude is given in
*                              WORLD(WCS(2)), longitude returned in
*                              WORLD(WCS(1)).
*      VSPAN    D(2)     Solution interval for the unknown celestial
*                        coordinate, in degrees.
*      VSTEP    D        Step size for solution search, in degrees.  If
*                        zero, a sensible, although perhaps non-optimal
*                        default will be used.
*      VITER    I        If a solution is not found then the step size
*                        will be halved and the search recommenced.
*                        VITER controls how many times the step size is
*                        halved.  The allowed range is 0 - 5.
*
*   Given and returned:
*      WORLD    D()      World coordinates.  WORLD(WCS(1)) and
*                        WORLD(WCS(2)) are the celestial longitude and
*                        latitude, in degrees.  Which is given and which
*                        returned depends on the value of MIXCEL.  All
*                        other elements are given.
*
*   Given:
*      CRVAL    D()      Coordinate reference values corresponding to
*                        the FITS CRVALn header cards.
*
*   Given and returned:
*      CEL      D(10)    Spherical coordinate transformation parameters
*                        (usage is described in the prologue to
*                        "cel.f").
*
*   Returned:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*
*   Given and returned:
*      PRJ      D(0:20)  Projection parameters (usage is described in
*                        the prologue to "proj.f").
*
*   Returned:
*      IMGCRD   D()      Image coordinate.  IMGCRD(WCS(1)) and
*                        IMGCRD(WCS(2)) are the projected x-, and
*                        y-coordinates, in "degrees".
*
*   Given and returned:
*      LIN      D()      Linear transformation parameters (usage is
*                        described in the prologue to "lin.f").
*
*   Given and returned:
*      PIXCRD   D()      Pixel coordinate.  The element indicated by
*                        MIXPIX is given and the remaining elements are
*                        returned.
*
*   Returned:
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid coordinate transformation
*                              parameters.
*                           2: Invalid projection parameters.
*                           3: Coordinate transformation error.
*                           4: Invalid linear transformation parameters.
*                           5: No solution found in the specified
*                              interval.
*
*
*   Notes
*   -----
*    1) The CTYPEn must in be upper case and there must be 0 or 1 pair of
*       matched celestial axis types.
*
*    2) Elements of the CRVAL array which correspond to celestial axes
*       are ignored, the reference coordinate values in CEL(1) and CEL(2)
*       are the ones used.
*
*
*   WCS indexing parameters
*   -----------------------
*   The WCS array contains indexes derived from the CTYPEn:
*
*      WCS(0)
*         This is an index into an internal array of WCS projection
*         types.  It must be set to zero when any of the CTYPE() are set
*         or changed.  This signals the initialization routine, WCSSET,
*         to recompute the indices.  WCS(0) is set to 999 if there is no
*         celestial axis pair in the CTYPEn.
*
*      WCS(1:2)
*         Indices into the IMGCRD, and WORLD arrays as described above.
*         These may also serve as indices for the celestial longitude
*         and latitude axes in the PIXCRD array provided that the PC
*         matrix does not transpose axes.
*
*      WCS(3)
*         Index into the PIXCRD array for the CUBEFACE axis.  This is
*         optionally used for the quadcube projections where each cube
*         face is stored on a separate axis.
*
*
*   WCSMIX algorithm
*   ----------------
*      Initially the specified solution interval is checked to see if
*      it's a "crossing" interval.  If it isn't, a search is made for a
*      crossing solution by iterating on the unknown celestial
*      coordinate starting at the upper limit of the solution interval
*      and decrementing by the specified step size.  A crossing is
*      indicated if the trial value of the projected coordinate steps
*      through the value specified.  If a crossing interval is found
*      then the solution is determined by a modified form of "regula
*      falsi" division of the crossing interval.  If no crossing
*      interval was found within the specified solution interval then a
*      search is made for a "non-crossing" solution as may arise from a
*      point of tangency.  The process is complicated by having to make
*      allowance for the discontinuities that occur in all map
*      projections.
*
*      Once one solution has been determined others may be found by
*      subsequent invokations of WCSMIX with suitably restricted
*      solution intervals.
*
*      Note that there is a circumstance where the problem posed to
*      WCSMIX is ill-conditioned and it may fail to find a valid
*      solution where one does exist.  This arises when the solution
*      point lies at a native pole of a projection in which the pole is
*      represented as a finite interval.
*
*      Because of its generality WCSMIX is very compute-intensive.  For
*      compute-limited applications more efficient special-case solvers
*      could be written for simple projections, for example non-oblique
*      cylindrical projections.
*
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: wcs.f,v 2.5 1996/09/10 06:49:58 mcalabre Exp $
*=======================================================================
      SUBROUTINE WCSSET (NAXIS, CTYPE, WCS, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J, K, NAXIS, WCS(0:3)
      CHARACTER CTYPE(*)*8, PCODE*3, PCODES*99, REQUIR*8

      DATA PCODES(01:40) /'AZP TAN SIN STG ARC ZPN ZEA AIR CYP CAR '/
      DATA PCODES(41:80) /'MER CEA COP COD COE COO BON PCO GLS PAR '/
      DATA PCODES(81:99) /'AIT MOL CSC QSC TSC'/
*-----------------------------------------------------------------------
      PCODE  = ' '
      REQUIR = ' '
      WCS(1) = 0
      WCS(2) = 0
      WCS(3) = 0

      DO 10 J = 1, NAXIS
         IF (CTYPE(J)(5:5).NE.'-') GO TO 10

*        Got an axis qualifier, is it a recognized WCS projection?
         IF (INDEX(PCODES,CTYPE(J)(6:8)).EQ.0) THEN
            IF (CTYPE(J).EQ.'CUBEFACE') THEN
               IF (WCS(3).EQ.0) THEN
                  WCS(3) = J
               ELSE
*                 Multiple CUBEFACE axes!
                  IERR = 1
                  RETURN
               END IF
            END IF

            GO TO 10
         END IF

*        Parse the celestial axis type.
         IF (PCODE.EQ.' ') THEN
            PCODE = CTYPE(J)(6:8)

            IF (CTYPE(J)(1:4).EQ.'RA--') THEN
               WCS(1) = J
               K = 2
               REQUIR = 'DEC--' // PCODE
            ELSE IF (CTYPE(J)(1:4).EQ.'DEC-') THEN
               WCS(2) = J
               K = 1
               REQUIR = 'RA---' // PCODE
            ELSE IF (CTYPE(J)(2:4).EQ.'LON') THEN
               WCS(1) = J
               K = 2
               REQUIR = CTYPE(J)(1:1) // 'LAT-' // PCODE
            ELSE IF (CTYPE(J)(2:4).EQ.'LAT') THEN
               WCS(2) = J
               K = 1
               REQUIR = CTYPE(J)(1:1) // 'LON-' // PCODE
            ELSE
*              Unrecognized celestial type.
               IERR = 1
               RETURN
            END IF
         ELSE
            IF (CTYPE(J).NE.REQUIR) THEN
*              Inconsistent projection types.
               IERR = 1
               RETURN
            END IF

            WCS(K) = J
            REQUIR = ' '
         END IF
 10   CONTINUE

      IF (REQUIR.NE.' ') THEN
*        Unmatched celestial axis.
         IERR = 1
         RETURN
      END IF

      IF (PCODE.NE.' ') THEN
         WCS(0) = (INDEX(PCODES,PCODE) - 1)/4 + 1
      ELSE
*        Signal for no celestial axis pair.
         WCS(0) = 999
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *   IMGCRD, LIN, PIXCRD, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J, NAXIS, WCS(0:3)
      DOUBLE PRECISION CEL(10), CRVAL(*), IMGCRD(*), LAT, LIN(*), LNG,
     *          OFFSET, PHI, PIXCRD(*), PRJ(0:20), THETA, WORLD(*), X, Y
      CHARACTER CTYPE(*)*8, PCODE*3, PCODES(25)*3

      DOUBLE PRECISION PI
      PARAMETER (PI = 3.141592653589793238462643D0)

      DATA PCODES /'AZP', 'TAN', 'SIN', 'STG', 'ARC',
     *             'ZPN', 'ZEA', 'AIR', 'CYP', 'CAR',
     *             'MER', 'CEA', 'COP', 'COD', 'COE',
     *             'COO', 'BON', 'PCO', 'GLS', 'PAR',
     *             'AIT', 'MOL', 'CSC', 'QSC', 'TSC'/
*-----------------------------------------------------------------------
*     Number of image axes.
      NAXIS = NINT(LIN(3))

*     Initialize if required.
      IF (WCS(0).LE.0) THEN
         CALL WCSSET (NAXIS, CTYPE, WCS, IERR)
         IF (IERR.NE.0) RETURN
      END IF

*     Convert to relative physical coordinates.
      DO 10 J = 1, NAXIS
         IF (J.EQ.WCS(1)) GO TO 10
         IF (J.EQ.WCS(2)) GO TO 10
         IMGCRD(J) = WORLD(J) - CRVAL(J)
 10   CONTINUE

      IF (WCS(0).NE.999) THEN
*        Compute projected coordinates.
         LNG = WORLD(WCS(1))
         LAT = WORLD(WCS(2))

         PCODE = PCODES(WCS(0))
         CALL CELFWD (PCODE, LNG, LAT, CEL, PHI, THETA, PRJ, X, Y, IERR)
         IF (IERR.NE.0) RETURN

         IMGCRD(WCS(1)) = X
         IMGCRD(WCS(2)) = Y

*        Do we have a CUBEFACE axis?
         IF (WCS(3).NE.0) THEN
*           Separation between faces.
            IF (PRJ(10).EQ.0D0) THEN
               OFFSET = 45D0
            ELSE
               OFFSET = PRJ(10)*PI/4D0
            END IF
   
*           Stack faces in a cube.
            IF (IMGCRD(WCS(2)).LT.OFFSET) THEN
               IMGCRD(WCS(2)) = IMGCRD(WCS(2)) + OFFSET
               IMGCRD(WCS(3)) = 5D0
            ELSE IF (IMGCRD(WCS(1)).LT.OFFSET*3) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) + OFFSET*3
               IMGCRD(WCS(3)) = 4D0
            ELSE IF (IMGCRD(WCS(1)).LT.OFFSET*2) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) + OFFSET*2
               IMGCRD(WCS(3)) = 3D0
            ELSE IF (IMGCRD(WCS(1)).LT.OFFSET) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) + OFFSET
               IMGCRD(WCS(3)) = 2D0
            ELSE IF (IMGCRD(WCS(2)).GT.OFFSET) THEN
               IMGCRD(WCS(2)) = IMGCRD(WCS(2)) - OFFSET
               IMGCRD(WCS(3)) = 0D0
            ELSE
               IMGCRD(WCS(3)) = 1D0
            END IF
         END IF
      END IF

*     Apply forward linear transformation.
      CALL LINFWD (IMGCRD, LIN, PIXCRD, IERR)
      IF (IERR.NE.0) THEN
         IERR = 4
         RETURN
      END IF

      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE WCSREV (CTYPE, WCS, PIXCRD, LIN, IMGCRD, PRJ, PHI,
     *   THETA, CRVAL, CEL, WORLD, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, FACE, J, NAXIS, WCS(0:3)
      DOUBLE PRECISION CEL(10), CRVAL(*), IMGCRD(*), LIN(*), LAT, LNG,
     *          OFFSET, PHI, PIXCRD(*), PRJ(0:20), THETA, WORLD(*), X,
     *          Y
      CHARACTER CTYPE(*)*8, PCODE*3, PCODES(25)*3

      DOUBLE PRECISION PI
      PARAMETER (PI = 3.141592653589793238462643D0)

      DATA PCODES /'AZP', 'TAN', 'SIN', 'STG', 'ARC',
     *             'ZPN', 'ZEA', 'AIR', 'CYP', 'CAR',
     *             'MER', 'CEA', 'COP', 'COD', 'COE',
     *             'COO', 'BON', 'PCO', 'GLS', 'PAR',
     *             'AIT', 'MOL', 'CSC', 'QSC', 'TSC'/
*-----------------------------------------------------------------------
*     Number of image axes.
      NAXIS = NINT(LIN(3))

*     Initialize if required.
      IF (WCS(0).LE.0) THEN
         CALL WCSSET (NAXIS, CTYPE, WCS, IERR)
         IF (IERR.NE.0) RETURN
      END IF

*     Apply reverse linear transformation.
      CALL LINREV (PIXCRD, LIN, IMGCRD, IERR)
      IF (IERR.NE.0) THEN
         IERR = 4
         RETURN
      END IF

*     Convert to world coordinates.
      DO 10 J = 1, NAXIS
         IF (J.EQ.WCS(1)) GO TO 10
         IF (J.EQ.WCS(2)) GO TO 10
         WORLD(J) = IMGCRD(J) + CRVAL(J)
 10   CONTINUE

      IF (WCS(0).NE.999) THEN
*        Do we have a CUBEFACE axis?
         IF (WCS(3).NE.0) THEN
            FACE = NINT(IMGCRD(WCS(3)))
            IF (ABS(IMGCRD(WCS(3))-FACE).GT.1D-10) THEN
               IERR = 3
               RETURN
            END IF

*           Separation between faces.
            IF (PRJ(10).EQ.0D0) THEN
               OFFSET = 45D0
            ELSE
               OFFSET = PRJ(10)*PI/4D0
            END IF

*           Lay out faces in a plane.
            IF (FACE.EQ.0) THEN
               IMGCRD(WCS(2)) = IMGCRD(WCS(2)) + OFFSET
            ELSE IF (FACE.EQ.1) THEN
*              Nothing.
            ELSE IF (FACE.EQ.2) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) - OFFSET
            ELSE IF (FACE.EQ.3) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) - OFFSET*2
            ELSE IF (FACE.EQ.4) THEN
               IMGCRD(WCS(1)) = IMGCRD(WCS(1)) - OFFSET*3
            ELSE IF (FACE.EQ.5) THEN
               IMGCRD(WCS(2)) = IMGCRD(WCS(2)) - OFFSET
            ELSE
               IERR = 3
               RETURN
            END IF
         END IF

*        Compute celestial coordinates.
         X = IMGCRD(WCS(1))
         Y = IMGCRD(WCS(2))

         PCODE = PCODES(WCS(0))
         CALL CELREV (PCODE, X, Y, PRJ, PHI, THETA, CEL, LNG, LAT, IERR)
         IF (IERR.NE.0) RETURN

         WORLD(WCS(1)) = LNG
         WORLD(WCS(2)) = LAT
      END IF

      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE WCSMIX (CTYPE, WCS, MIXPIX, MIXCEL, VSPAN, VSTEP,
     *   VITER, WORLD, CRVAL, CEL, PHI, THETA, PRJ, IMGCRD, LIN, PIXCRD,
     *   IERR)
*-----------------------------------------------------------------------
      INTEGER   CROSSED, IERR, ISTEP, ITER, MIXCEL, MIXPIX, NITER,
     *          RETRY, VITER, WCS(0:3)
      DOUBLE PRECISION CEL(10), CRVAL(*), D, D0, D0M, D1, D1M, DABS,
     *          DMIN, DX, IMGCRD(*), LAMBDA, LAT, LAT0, LAT0M, LAT1,
     *          LAT1M, LIN(*), LMIN, LNG, LNG0, LNG0M, LNG1, LNG1M, PHI,
     *          PIXCRD(*), PIXMIX, PRJ(0:20), SPAN(0:1), STEP, THETA,
     *          TOL, VSPAN(0:1), VSTEP, WORLD(*)
      CHARACTER CTYPE(*)*8

      PARAMETER (NITER = 60)
      PARAMETER (TOL = 1D-10)
*-----------------------------------------------------------------------
*     Check VSPAN.
      IF (VSPAN(0).LE.VSPAN(1)) THEN
         SPAN(0) = VSPAN(0)
         SPAN(1) = VSPAN(1)
      ELSE
*        Swap them.
         SPAN(0) = VSPAN(1)
         SPAN(1) = VSPAN(0)
      END IF

*     Check VSTEP.
      STEP = ABS(VSTEP)
      IF (STEP.EQ.0D0) THEN
         STEP = (SPAN(1) - SPAN(0))/10D0
         IF (STEP.GT.1D0 .OR. STEP.EQ.0D0) STEP = 1D0
      END IF

*     Check VITER.
      IF (VITER.LT.0) THEN
         VITER = 0
      ELSE IF (VITER.GT.5) THEN
         VITER = 5
      END IF

*     Given pixel element.
      PIXMIX = PIXCRD(MIXPIX)

*     Iterate on the step size.
      DO 110 ISTEP = 0, VITER
         IF (ISTEP.GT.0) STEP = STEP/2D0

*        Iterate on the sky coordinate between the specified range.
         IF (MIXCEL.EQ.1) THEN
*           Celestial longitude is given.

*           Check whether the solution interval is a crossing interval.
            LAT0 = SPAN(0)
            WORLD(WCS(2)) = LAT0
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *         IMGCRD, LIN, PIXCRD, IERR)
            IF (IERR.NE.0) RETURN
            D0 = PIXCRD(MIXPIX) - PIXMIX

            DABS = ABS(D0)
            IF (DABS.LT.TOL) RETURN

            LAT1 = SPAN(1)
            WORLD(WCS(2)) = LAT1
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *         IMGCRD, LIN, PIXCRD, IERR)
            IF (IERR.NE.0) RETURN
            D1 = PIXCRD(MIXPIX) - PIXMIX

            DABS = ABS(D1)
            IF (DABS.LT.TOL) RETURN

            LMIN = LAT1
            DMIN = DABS

*           Check for a crossing point.
            IF (SIGN(1D0,D0).NE.SIGN(1D0,D1)) THEN
               CROSSED = 1
               DX = D1
            ELSE
               CROSSED = 0
               LAT0 = SPAN(1)
            END IF

            DO 50 RETRY = 1, 4
*              Refine the solution interval.
 10            IF (LAT0.GT.SPAN(0)) THEN
                  LAT0 = LAT0 - STEP
                  IF (LAT0.LT.SPAN(0)) LAT0 = SPAN(0)
                  WORLD(WCS(2)) = LAT0
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D0 = PIXCRD(MIXPIX) - PIXMIX

*                 Check for a solution.
                  DABS = ABS(D0)
                  IF (DABS.LT.TOL) RETURN

*                 Record the point of closest approach.
                  IF (DABS.LT.DMIN) THEN
                     LMIN = LAT0
                     DMIN = DABS
                  END IF

*                 Check for a crossing point.
                  IF (SIGN(1D0,D0).NE.SIGN(1D0,D1)) THEN
                     CROSSED = 2
                     DX = D0
                     GO TO 20
                  END IF

*                 Advance to the next subinterval.
                  LAT1 = LAT0
                  D1 = D0
                  GO TO 10
               END IF

 20            IF (CROSSED.NE.0) THEN
*                 A crossing point was found.
                  DO 30 ITER = 1, NITER
*                    Use regula falsi division of the interval.
                     LAMBDA = D0/(D0-D1)
                     IF (LAMBDA.LT.0.1D0) THEN
                        LAMBDA = 0.1D0
                     ELSE IF (LAMBDA.GT.0.9D0) THEN
                        LAMBDA = 0.9
                     END IF

                     LAT = LAT0 + LAMBDA*(LAT1 - LAT0)
                     WORLD(WCS(2)) = LAT
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D = PIXCRD(MIXPIX) - PIXMIX

*                    Check for a solution.
                     DABS = ABS(D)
                     IF (DABS.LT.TOL) RETURN

*                    Record the point of closest approach.
                     IF (DABS.LT.DMIN) THEN
                        LMIN = LAT
                        DMIN = DABS
                     END IF

                     IF (SIGN(1D0,D0).EQ.SIGN(1D0,D)) THEN
                        LAT0 = LAT
                        D0 = D
                     ELSE
                        LAT1 = LAT
                        D1 = D
                     END IF
 30               CONTINUE

*                 No convergence, must have been a discontinuity.
                  IF (CROSSED.EQ.1) LAT0 = SPAN(1)
                  LAT1 = LAT0
                  D1 = DX
                  CROSSED = 0

               ELSE
*                 No crossing point; look for a tangent point.
                  IF (LMIN.EQ.SPAN(0)) GO TO 110
                  IF (LMIN.EQ.SPAN(1)) GO TO 110

                  LAT  = LMIN
                  LAT0 = LAT - STEP
                  IF (LAT0.LT.SPAN(0)) LAT0 = SPAN(0)
                  LAT1 = LAT + STEP
                  IF (LAT1.GT.SPAN(1)) LAT1 = SPAN(1)

                  WORLD(WCS(2)) = LAT0
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D0 = ABS(PIXCRD(MIXPIX) - PIXMIX)

                  D  = DMIN

                  WORLD(WCS(2)) = LAT1
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D1 = ABS(PIXCRD(MIXPIX) - PIXMIX)

                  DO 40 ITER = 1, NITER
                     LAT0M = (LAT0 + LAT)/2D0
                     WORLD(WCS(2)) = LAT0M
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D0M = ABS(PIXCRD(MIXPIX) - PIXMIX)

                     IF (D0M.LT.TOL) RETURN

                     LAT1M = (LAT1 + LAT)/2D0
                     WORLD(WCS(2)) = LAT1M
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D1M = ABS(PIXCRD(MIXPIX) - PIXMIX)

                     IF (D1M.LT.TOL) RETURN

                     IF (D0M.LT.D .AND. D0M.LE.D1M) THEN
                        LAT1 = LAT
                        D1   = D
                        LAT  = LAT0M
                        D    = D0M
                     ELSE IF (D1M.LT.D) THEN
                        LAT0 = LAT
                        D0   = D
                        LAT  = LAT1M
                        D    = D1M
                     ELSE
                        LAT0 = LAT0M
                        D0   = D0M
                        LAT1 = LAT1M
                        D1   = D1M
                     END IF
 40               CONTINUE
               END IF
 50         CONTINUE

         ELSE
*           Celestial latitude is given.

*           Check whether the solution interval is a crossing interval.
            LNG0 = SPAN(0)
            WORLD(WCS(1)) = LNG0
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *         IMGCRD, LIN, PIXCRD, IERR)
            IF (IERR.NE.0) RETURN
            D0 = PIXCRD(MIXPIX) - PIXMIX

            DABS = ABS(D0)
            IF (DABS.LT.TOL) RETURN

            LNG1 = SPAN(1)
            WORLD(WCS(1)) = LNG1
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *         IMGCRD, LIN, PIXCRD, IERR)
            IF (IERR.NE.0) RETURN
            D1 = PIXCRD(MIXPIX) - PIXMIX

            DABS = ABS(D1)
            IF (DABS.LT.TOL) RETURN

            LMIN = LNG1
            DMIN = DABS

*           Check for a crossing point.
            IF (SIGN(1D0,D0).NE.SIGN(1D0,D1)) THEN
               CROSSED = 1
               DX = D1
            ELSE
               CROSSED = 0
               LNG0 = SPAN(1)
            END IF

            DO 100 RETRY = 1, 4
*              Refine the solution interval.
 60            IF (LNG0.GT.SPAN(0)) THEN
                  LNG0 = LNG0 - STEP
                  IF (LNG0.LT.SPAN(0)) LNG0 = SPAN(0)
                  WORLD(WCS(1)) = LNG0
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D0 = PIXCRD(MIXPIX) - PIXMIX

*                 Check for a solution.
                  DABS = ABS(D0)
                  IF (DABS.LT.TOL) RETURN

*                 Record the point of closest approach.
                  IF (DABS.LT.DMIN) THEN
                     LMIN = LNG0
                     DMIN = DABS
                  END IF

*                 Check for a crossing point.
                  IF (SIGN(1D0,D0).NE.SIGN(1D0,D1)) THEN
                     CROSSED = 2
                     DX = D0
                     GO TO 70
                  END IF

*                 Advance to the next subinterval.
                  LNG1 = LNG0
                  D1 = D0
                  GO TO 60
               END IF

 70            IF (CROSSED.GT.0) THEN
*                 A crossing point was found.
                  DO 80 ITER = 1, NITER
*                    Use regula falsi division of the interval.
                     LAMBDA = D0/(D0-D1)
                     IF (LAMBDA.LT.0.1D0) THEN
                        LAMBDA = 0.1D0
                     ELSE IF (LAMBDA.GT.0.9D0) THEN
                        LAMBDA = 0.9D0
                     END IF

                     LNG = LNG0 + LAMBDA*(LNG1 - LNG0)
                     WORLD(WCS(1)) = LNG
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D = PIXCRD(MIXPIX) - PIXMIX

*                    Check for a solution.
                     DABS = ABS(D)
                     IF (DABS.LT.TOL) RETURN

*                    Record the point of closest approach.
                     IF (DABS.LT.DMIN) THEN
                        LMIN = LNG
                        DMIN = DABS
                     END IF

                     IF (SIGN(1D0,D0).EQ.SIGN(1D0,D)) THEN
                        LNG0 = LNG
                        D0 = D
                     ELSE
                        LNG1 = LNG
                        D1 = D
                     END IF
 80               CONTINUE

*                 No convergence, must have been a discontinuity.
                  IF (CROSSED.EQ.1) LNG0 = SPAN(1)
                  LNG1 = LNG0
                  D1 = DX
                  CROSSED = 0

               ELSE
*                 No crossing point; look for a tangent point.
                  IF (LMIN.EQ.SPAN(0)) GO TO 110
                  IF (LMIN.EQ.SPAN(1)) GO TO 110

                  LNG  = LMIN
                  LNG0 = LNG - STEP
                  IF (LNG0.LT.SPAN(0)) LNG0 = SPAN(0)
                  LNG1 = LNG + STEP
                  IF (LNG1.GT.SPAN(1)) LNG1 = SPAN(1)

                  WORLD(WCS(1)) = LNG0
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D0 = ABS(PIXCRD(MIXPIX) - PIXMIX)

                  D  = DMIN

                  WORLD(WCS(1)) = LNG1
                  CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *               THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                  IF (IERR.NE.0) RETURN
                  D1 = ABS(PIXCRD(MIXPIX) - PIXMIX)

                  DO 90 ITER = 1, NITER
                     LNG0M = (LNG0 + LNG)/2D0
                     WORLD(WCS(1)) = LNG0M
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D0M = ABS(PIXCRD(MIXPIX) - PIXMIX)

                     IF (D0M.LT.TOL) RETURN

                     LNG1M = (LNG1 + LNG)/2D0
                     WORLD(WCS(1)) = LNG1M
                     CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI,
     *                  THETA, PRJ, IMGCRD, LIN, PIXCRD, IERR)
                     IF (IERR.NE.0) RETURN
                     D1M = ABS(PIXCRD(MIXPIX) - PIXMIX)

                     IF (D1M.LT.TOL) RETURN

                     IF (D0M.LT.D .AND. D0M.LE.D1M) THEN
                        LNG1 = LNG
                        D1   = D
                        LNG  = LNG0M
                        D    = D0M
                     ELSE IF (D1M.LT.D) THEN
                        LNG0 = LNG
                        D0   = D
                        LNG  = LNG1M
                        D    = D1M
                     ELSE
                        LNG0 = LNG0M
                        D0   = D0M
                        LNG1 = LNG1M
                        D1   = D1M
                     END IF
 90               CONTINUE
               END IF
 100        CONTINUE
         END IF
 110  CONTINUE

*     No solution.
      IERR = 5
      RETURN
      END
