*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2000, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
      PROGRAM TWCS2
*-----------------------------------------------------------------------
*
*   TWCS2 tests WCSMIX for closure on the 1 degree celestial grid for
*   a number of selected projections.  Points with good solutions are
*   marked with a white dot on a graphical display of the projection
*   while bad solutions are flagged with a red circle.
*
*   $Id: twcs2.f,v 2.11 2000/05/10 04:51:30 mcalabre Exp $
*-----------------------------------------------------------------------
*     Set maximum number of axes (MAXIS) and number of axes (NAXIS).
      INTEGER   M, N
      PARAMETER (M = 10, N = 4)

*     Force alignment.
      COMMON /DUMMY/ LIN, MAXIS, NAXIS, CRPIX, PC, CDELT, PIXIMG, IMGPIX

      INTEGER   I, IERR, J
      DOUBLE PRECISION CDELT(M), CRPIX(M), CRVAL(M), CEL0(10), CELC(10),
     *          CELP(10), IMGPIX(M,M), LATC, LIN, MAXIS, NAXIS,
     *          NTV0(10), NTVC(10), NTVP(10), PC(M,M), PIXIMG(M,M),
     *          PRJ(0:20)
      CHARACTER CTYPE(M)*8

      DOUBLE PRECISION TOL
      PARAMETER (TOL = 1D-9)

      DATA (CTYPE(I), I=1,N)
     *           /'FREQ    ', 'XLAT--xxx', 'TIME    ', 'XLON--xxx'/
      DATA (CRPIX(I), I=1,N)
     *           / 0.0D0,  0.0D0,  0.0D0,  0.0D0/
      DATA ((PC(I,J),I=1,N),J=1,N)
     *           / 1.1D0,    0D0,    0D0,    0D0,
     *               0D0,  1.0D0,    0D0,  0.1D0,
     *               0D0,    0D0,  1.0D0,    0D0,
     *               0D0,  0.2D0,    0D0,  1.0D0/
      DATA (CDELT(I), I=1,N)
     *           / 1.0D0,  1.0D0,  1.0D0, -1.0D0/
      DATA (CRVAL(I), I=1,N)
     *           / 408D6,  0.0D0,   -2D3,  0.0D0/
*-----------------------------------------------------------------------
*     Uncomment the following two lines to raise SIGFPE on floating
*     point exceptions for the Sun FORTRAN compiler.  This signal can
*     be caught within 'dbx' by issuing the command "catch FPE".
*#include <floatingpoint.h>
*     call ieee_handler ('set', 'common', SIGFPE_ABORT)

      WRITE (6, 5)
 5    FORMAT (/,'Testing WCSLIB wcsmix routine',/,
     *          '-----------------------------')

*     PGPLOT initialization.
      CALL PGBEG (0, '/xwindow', 1, 1)

*     Define pen colours.
      CALL PGSCR (0, 0.00, 0.00, 0.00)
      CALL PGSCR (1, 1.00, 1.00, 0.00)
      CALL PGSCR (2, 1.00, 1.00, 1.00)
      CALL PGSCR (3, 0.50, 0.50, 0.80)
      CALL PGSCR (4, 0.80, 0.50, 0.50)
      CALL PGSCR (5, 0.80, 0.80, 0.80)
      CALL PGSCR (6, 0.50, 0.50, 0.80)
      CALL PGSCR (7, 0.80, 0.50, 0.50)
      CALL PGSCR (8, 0.30, 0.50, 0.30)
      CALL PGSCR (9, 1.00, 0.75, 0.00)

      LIN   = 0D0
      MAXIS = DBLE(M)
      NAXIS = DBLE(N)

      DO 10 J = 0, 20
         PRJ(J) = 0D0
 10   CONTINUE

*     Latitude midway between the standard parallels for the conics.
      LATC = 60D0

*     Set reference angles for native grids; polar projections...
      NTVP(1)  =   0D0
      NTVP(2)  =  90D0
      NTVP(3)  = 999D0
      NTVP(4)  = 999D0

*     ...conic projections...
      NTVC(1)  =   0D0
      NTVC(2)  =  LATC
      NTVC(3)  = 999D0
      NTVC(4)  = 999D0

*     ...cylindrical and conventional projections.
      NTV0(1)  =   0D0
      NTV0(2)  =   0D0
      NTV0(3)  = 999D0
      NTV0(4)  = 999D0

*     Set reference angles for the celestial grids; polar projections...
      CELP(1) = 150D0
      CELP(2) = -30D0
      CELP(3) = 150D0
      CELP(4) = 999D0

*     Force CELP to be initialized since we want to use it now.
      CELP(5) = 0D0
      CALL CELSET ('ARC', CELP, PRJ, IERR)

*     Compute reference angles for the cylindrical and conic projections
*     so that they all use the same oblique celestial grid regardless of
*     the reference point; conic projections...
      CALL SPHREV (0D0, LATC, CELP(6), CELC(1), CELC(2), IERR)
      CALL SPHFWD (0D0, 90D0, CELP(6), CELC(3), CELC(4), IERR)

*     ...cylindrical and conventional projections.
      CALL SPHREV (0D0,  0D0, CELP(6), CEL0(1), CEL0(2), IERR)
      CALL SPHFWD (0D0, 90D0, CELP(6), CEL0(3), CEL0(4), IERR)

*     Note that we have 6 contexts (NTVP, NTVC, NTV0, CELP, CELC, and
*     CEL0).  Routines GRDPLT and MIXEX will each force these to be
*     initialized on every invokation since PCODE will differ each time.

*     ARC: zenithal/azimuthal equidistant.
      CTYPE(2)(6:8) = 'ARC'
      CTYPE(4)(6:8) = 'ARC'
      CALL GRDPLT (1, -190.0, 190.0, -190.0, 190.0, CTYPE, CRVAL, NTVP,
     *   CELP, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CELP, PRJ, LIN)

*     ZEA: zenithal/azimuthal equal area.
      CTYPE(2)(6:8) = 'ZEA'
      CTYPE(4)(6:8) = 'ZEA'
      CALL GRDPLT (1, -120.0, 120.0, -120.0, 120.0, CTYPE, CRVAL, NTVP,
     *   CELP, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CELP, PRJ, LIN)

*     CYP: cylindrical perspective.
      CTYPE(2)(6:8) = 'CYP'
      CTYPE(4)(6:8) = 'CYP'
      PRJ(1) = 3D0
      PRJ(2) = 0.8D0
      CALL GRDPLT (2, -170.0, 170.0, -170.0, 170.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     CAR: Cartesian.
      CTYPE(2)(6:8) = 'CAR'
      CTYPE(4)(6:8) = 'CAR'
      CALL GRDPLT (2, -210.0, 210.0, -210.0, 210.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     CEA: cylindrical equal area.
      CTYPE(2)(6:8) = 'CEA'
      CTYPE(4)(6:8) = 'CEA'
      PRJ(1) = 0.75D0
      CALL GRDPLT (2, -200.0, 200.0, -200.0, 200.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     COD: conic equidistant.
      CTYPE(2)(6:8) = 'COD'
      CTYPE(4)(6:8) = 'COD'
      PRJ(1) = LATC
      PRJ(2) = 15D0
      CALL GRDPLT (3, -200.0, 200.0, -180.0, 220.0, CTYPE, CRVAL, NTVC,
     *   CELC, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CELC, PRJ, LIN)

*     COE: conic equal area.
      CTYPE(2)(6:8) = 'COE'
      CTYPE(4)(6:8) = 'COE'
      PRJ(1) = LATC
      PRJ(2) = 15D0
      CALL GRDPLT (3, -140.0, 140.0, -120.0, 160.0, CTYPE, CRVAL, NTVC,
     *   CELC, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CELC, PRJ, LIN)

*     BON: Bonne's projection.
      CTYPE(2)(6:8) = 'BON'
      CTYPE(4)(6:8) = 'BON'
      PRJ(1) = 30D0
      CALL GRDPLT (4, -160.0, 160.0, -160.0, 160.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     PCO: polyconic.
      CTYPE(2)(6:8) = 'PCO'
      CTYPE(4)(6:8) = 'PCO'
      CALL GRDPLT (4, -190.0, 190.0, -190.0, 190.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     GLS: Sanson-Flamsteed (global sinusoid).
      CTYPE(2)(6:8) = 'GLS'
      CTYPE(4)(6:8) = 'GLS'
      CALL GRDPLT (4, -190.0, 190.0, -190.0, 190.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     PAR: parabolic.
      CTYPE(2)(6:8) = 'PAR'
      CTYPE(4)(6:8) = 'PAR'
      CALL GRDPLT (4, -190.0, 190.0, -190.0, 190.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     AIT: Hammer-Aitoff.
      CTYPE(2)(6:8) = 'AIT'
      CTYPE(4)(6:8) = 'AIT'
      CALL GRDPLT (4, -170.0, 170.0, -170.0, 170.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     MOL: Mollweide's projection.
      CTYPE(2)(6:8) = 'MOL'
      CTYPE(4)(6:8) = 'MOL'
      CALL GRDPLT (4, -170.0, 170.0, -170.0, 170.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     QSC: quadrilateralized spherical cube.
      CTYPE(2)(6:8) = 'QSC'
      CTYPE(4)(6:8) = 'QSC'
      CALL GRDPLT (5, -340.0, 80.0, -210.0, 210.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

*     TSC: tangential spherical cube.
      CTYPE(2)(6:8) = 'TSC'
      CTYPE(4)(6:8) = 'TSC'
      CALL GRDPLT (5, -340.0, 80.0, -210.0, 210.0, CTYPE, CRVAL, NTV0,
     *   CEL0, PRJ, LIN)
      CALL MIXEX (TOL, CTYPE, CRVAL, CEL0, PRJ, LIN)

      CALL PGEND ()

      END


*-----------------------------------------------------------------------
      SUBROUTINE GRDPLT (TYPE, IMIN, IMAX, JMIN, JMAX, CTYPE, CRVAL,
     *   NATIVE, CELEST, PRJ, LIN)
*-----------------------------------------------------------------------
      INTEGER   CI, IERR, ILAT, ILNG, K, TYPE, WCS(0:3)
      REAL      IMAX, IMIN, JMAX, JMIN, IR(512), JR(512)
      DOUBLE PRECISION CELEST(10), CRVAL(*), IMG(4), NATIVE(10), LAT,
     *          LIN(*), LNG, PHI, PIX(4), PRJ(0:20), STEP, THETA,
     *          WORLD(4)
      CHARACTER CTYPE(*)*8, PCODE*3, TEXT*80
*-----------------------------------------------------------------------
      WCS(0) = 0
      NATIVE(5) = 0D0
      CELEST(5) = 0D0
      PRJ(10) = 0D0
      PRJ(11) = 0D0

*     Define PGPLOT viewport.
      CALL PGENV (IMIN, IMAX, JMIN, JMAX, 1, -2)

*     Issue a dummy call to initialize data structures.
      WORLD(1) = CRVAL(1)
      WORLD(2) = CRVAL(2)
      WORLD(3) = CRVAL(3)
      WORLD(4) = CRVAL(4)
      CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CELEST, PHI, THETA, PRJ,
     *   IMG, LIN, PIX, IERR)
      PCODE = CTYPE(WCS(1))(6:8)

      IF (TYPE.EQ.5) THEN
*        Some sort of quad-cube projection.
         CALL PGSCI (8)

*        Draw the map boundary.
         IMG(1) = 0D0
         IMG(2) = 0D0
         IMG(3) = 0D0
         IMG(4) = 0D0

         IMG(WCS(1)) = -PRJ(12)
         IMG(WCS(2)) =  PRJ(12)
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(1) = PIX(WCS(1))
         JR(1) = PIX(WCS(2))

         IMG(WCS(1)) = -PRJ(12)
         IMG(WCS(2)) =  PRJ(12)*3D0
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(2) = PIX(WCS(1))
         JR(2) = PIX(WCS(2))

         IMG(WCS(1)) =  PRJ(12)
         IMG(WCS(2)) =  PRJ(12)*3D0
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(3) = PIX(WCS(1))
         JR(3) = PIX(WCS(2))

         IMG(WCS(1)) =  PRJ(12)
         IMG(WCS(2)) = -PRJ(12)*3D0
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(4) = PIX(WCS(1))
         JR(4) = PIX(WCS(2))

         IMG(WCS(1)) = -PRJ(12)
         IMG(WCS(2)) = -PRJ(12)*3D0
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(5) = PIX(WCS(1))
         JR(5) = PIX(WCS(2))

         IMG(WCS(1)) = -PRJ(12)
         IMG(WCS(2)) =  PRJ(12)
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(6) = PIX(WCS(1))
         JR(6) = PIX(WCS(2))

         IMG(WCS(1)) =  PRJ(12)*7D0
         IMG(WCS(2)) =  PRJ(12)
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(7) = PIX(WCS(1))
         JR(7) = PIX(WCS(2))

         IMG(WCS(1)) =  PRJ(12)*7D0
         IMG(WCS(2)) = -PRJ(12)
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(8) = PIX(WCS(1))
         JR(8) = PIX(WCS(2))

         IMG(WCS(1)) = -PRJ(12)
         IMG(WCS(2)) = -PRJ(12)
         CALL LINFWD (IMG, LIN, PIX, IERR)
         IR(9) = PIX(WCS(1))
         JR(9) = PIX(WCS(2))

         CALL PGLINE (9, IR, JR)
      END IF

*     Write a descriptive title.
      CALL PGSCI (1)
      TEXT = PCODE // ' projection - 15 degree graticule'
      WRITE (6, '(/,A)') TEXT
      CALL PGTEXT (IMIN, JMIN-10.0, TEXT)

      WRITE (TEXT, 10) CELEST(1), CELEST(2)
 10   FORMAT ('centered on celestial coordinates (',F6.2,',',F6.2,')')
      WRITE (6, '(A)') TEXT
      CALL PGTEXT (IMIN, JMIN-20.0, TEXT)

      WRITE (TEXT, 20) CELEST(3), CELEST(4)
 20   FORMAT ('with celestial pole at native coordinates (',F7.2,
     *   ',',F7.2,')')
      WRITE (6, '(A)') TEXT
      CALL PGTEXT (IMIN, JMIN-30.0, TEXT)


*     Draw the native coordinate grid faintly in the background.
      CALL PGSCI (8)

      IF (TYPE.EQ.4) THEN
         STEP = 10D0
      ELSE
         STEP = 15D0
      END IF

*     Draw native meridians of longitude.
      DO 40 ILNG = -180, 180, 15
         LNG = DBLE(ILNG)
         IF (ILNG.EQ.-180) LNG = -179.99D0
         IF (ILNG.EQ.+180) LNG = +179.99D0

         K = 0
         DO 30 ILAT = -90, 90
            LAT = DBLE(ILAT)

            WORLD(WCS(1)) = LNG
            WORLD(WCS(2)) = LAT
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, NATIVE, PHI, THETA,
     *         PRJ, IMG, LIN, PIX, IERR)
            IF (IERR.NE.0) GO TO 30

            IF (TYPE.EQ.5 .AND. K.GT.0) THEN
               IF (ABS(PIX(WCS(1))-IR(K)).GT.2.0 .OR.
     *             ABS(PIX(WCS(2))-JR(K)).GT.5.0) THEN
                  IF (K.GT.1) CALL PGLINE (K, IR, JR)
                  K = 0
               END IF
            END IF

            K = K + 1
            IR(K) = PIX(WCS(1))
            JR(K) = PIX(WCS(2))
 30      CONTINUE

         CALL PGLINE (K, IR, JR)
 40   CONTINUE

*     Draw native parallels of latitude.
      DO 60 ILAT = -90, 90, 15
         LAT = DBLE(ILAT)

         K = 0
         DO 50 ILNG = -180, 180
            LNG = DBLE(ILNG)
            IF (ILNG.EQ.-180) LNG = -179.99D0
            IF (ILNG.EQ.+180) LNG = +179.99D0

            WORLD(WCS(1)) = LNG
            WORLD(WCS(2)) = LAT
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, NATIVE, PHI, THETA,
     *         PRJ, IMG, LIN, PIX, IERR)
            IF (IERR.NE.0) GO TO 50

            IF (TYPE.EQ.5 .AND. K.GT.0) THEN
               IF (ABS(PIX(WCS(1))-IR(K)).GT.2.0 .OR.
     *             ABS(PIX(WCS(2))-JR(K)).GT.5.0) THEN
                  IF (K.GT.1) CALL PGLINE (K, IR, JR)
                  K = 0
               END IF
            END IF

            K = K + 1
            IR(K) = PIX(WCS(1))
            JR(K) = PIX(WCS(2))
 50      CONTINUE

         CALL PGLINE (K, IR, JR)
 60   CONTINUE


*     Draw a colour-coded celestial coordinate grid.
      CI = 1

*     Draw celestial meridians of longitude.
      DO 80 ILNG = -180, 180, 15
         LNG = DBLE(ILNG)

         CI = CI + 1
         IF (CI.GT.7) CI = 2
         IF (ILNG.NE.0) THEN
            CALL PGSCI (CI)
         ELSE
            CALL PGSCI (1)
         END IF

         K = 0
         DO 70 ILAT = -90, 90
            LAT = DBLE(ILAT)

            WORLD(WCS(1)) = LNG
            WORLD(WCS(2)) = LAT
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CELEST, PHI, THETA,
     *         PRJ, IMG, LIN, PIX, IERR)
            IF (IERR.NE.0) GO TO 70

*           Test for discontinuities.
            IF (K.GT.0) THEN
               IF (ABS(PIX(WCS(1))-IR(K)).GT.STEP .OR.
     *             ABS(PIX(WCS(2))-JR(K)).GT.STEP) THEN
                  IF (K.GT.1) CALL PGLINE (K, IR, JR)
                  K = 0
               END IF
            END IF

            K = K + 1
            IR(K) = PIX(WCS(1))
            JR(K) = PIX(WCS(2))
 70      CONTINUE

         CALL PGLINE (K, IR, JR)
 80   CONTINUE

*     Draw celestial parallels of latitude.
      CI = 1
      DO 100 ILAT = -90, 90, 15
         LAT = DBLE(ILAT)

         CI = CI + 1
         IF (CI.GT.7) CI = 2
         IF (ILAT.NE.0) THEN
            CALL PGSCI (CI)
         ELSE
            CALL PGSCI (1)
         END IF

         K = 0
         DO 90 ILNG = -180, 180
            LNG = DBLE(ILNG)

            WORLD(WCS(1)) = LNG
            WORLD(WCS(2)) = LAT
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CELEST, PHI, THETA,
     *         PRJ, IMG, LIN, PIX, IERR)
            IF (IERR.NE.0) GO TO 90

*           Test for discontinuities.
            IF (K.GT.0) THEN
               IF (ABS(PIX(WCS(1))-IR(K)).GT.STEP .OR.
     *             ABS(PIX(WCS(2))-JR(K)).GT.STEP) THEN
                  IF (K.GT.1) CALL PGLINE (K, IR, JR)
                  K = 0
               END IF
            END IF

            K = K + 1
            IR(K) = PIX(WCS(1))
            JR(K) = PIX(WCS(2))
 90      CONTINUE

         CALL PGLINE (K, IR, JR)
 100  CONTINUE

      CALL PGSCI (2)

      RETURN
      END


      SUBROUTINE MIXEX (TOL, CTYPE, CRVAL, CEL, PRJ, LIN)
*-----------------------------------------------------------------------
*   MIXEX tests WCSMIX.
*
*   Given:
*      TOL      D        Reporting tolerance, degrees.
*      CTYPE    C()*8    Coordinate axis types.
*      CRVAL    D()      Coordinate reference values.
*
*   Given and returned:
*      CEL      D(10)    Coordinate transformation parameters.
*      PRJ      D(0:20)  Projection parameters.
*      LIN      D()      Linear transformation parameters.
*-----------------------------------------------------------------------
      INTEGER   DOID, IERR, K, LAT, LNG, WCS(0:3)
      REAL      IPT(1), JPT(1)
      DOUBLE PRECISION CEL(10), CRVAL(*), IMG(4), LAT1, LATSPN(2),
     *          LIN(*), LNGSPN(2), LNG1, PHI, PIX1(4), PIX2(4),
     *          PIX3(4), PIXLAT, PIXLNG, PRJ(0:20), THETA, TOL, WORLD(4)
      CHARACTER CTYPE(*)*8, PCODE*3
*-----------------------------------------------------------------------
      WCS(0)  = 0
      CEL(5)  = 0D0
      PRJ(10) = 0D0
      PRJ(11) = 0D0

*     Find the projection code.
      CALL WCSSET (NINT(LIN(3)), CTYPE, WCS, IERR)
      PCODE = CTYPE(WCS(1))(6:8)

      WRITE (6, 10) PCODE, TOL
 10   FORMAT ('Testing ',A,'; reporting tolerance',1PG8.1,' deg.')

      WORLD(1) = 0D0
      WORLD(2) = 0D0
      WORLD(3) = 0D0
      WORLD(4) = 0D0

      DO 80 LAT = 90, -90, -1
         LAT1 = DBLE(LAT)

         DO 70 LNG = -180, 180
            LNG1 = DBLE(LNG)

            WORLD(WCS(1)) = LNG1
            WORLD(WCS(2)) = LAT1
            CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA, PRJ,
     *         IMG, LIN, PIX1, IERR)
            IF (IERR.NE.0) THEN
               WRITE (6, 20) PCODE, LNG1, LAT1, IERR
 20            FORMAT (A3,': LNG1 =',F20.15,'  LAT1 =',F20.15,
     *            '  error',I3)
               GO TO 70
            END IF

            PIXLNG = PIX1(WCS(1))
            PIXLAT = PIX1(WCS(2))

            IPT(1) = PIXLNG
            JPT(1) = PIXLAT
            CALL PGPT (1, IPT, JPT, -1)

            LNGSPN(1) = LNG1 - 9.3D0
            IF (LNGSPN(1).LT.-180D0) LNGSPN(1) = -180D0
            LNGSPN(2) = LNG1 + 4.1D0
            IF (LNGSPN(2).GT. 180D0) LNGSPN(2) =  180D0
            LATSPN(1) = LAT1 - 3.7D0
            IF (LATSPN(1).LT. -90D0) LATSPN(1) =  -90D0
            LATSPN(2) = LAT1 + 7.2D0
            IF (LATSPN(2).GT.  90D0) LATSPN(2) =   90D0

            DOID = 1

            PIX2(WCS(1)) = PIXLNG
            CALL WCSMIX (CTYPE, WCS, WCS(1), 1, LATSPN, 1D0, 0, WORLD,
     *         CRVAL, CEL, PHI, THETA, PRJ, IMG, LIN, PIX2, IERR)
            IF (IERR.NE.0) THEN
               CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
               WRITE (6, '(A,I2)') '  A: WCSMIX error', IERR
            ELSE
               CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA,
     *            PRJ, IMG, LIN, PIX3, IERR)
               IF (IERR.NE.0) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, '(A,I2)') '  A: WCSFWD error', IERR
               ELSE IF (ABS(PIX3(WCS(1))-PIXLNG).GT.TOL .AND.
     *                 (ABS(WORLD(WCS(2))- LAT1).GT.TOL .OR.
     *                  ABS(PIX2(WCS(2))-PIXLAT).GT.TOL)) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, 30) WORLD(WCS(2)), PIX2(WCS(2))
 30               FORMAT ('  A: LAT2 =',F20.15,'    J2 =',F20.15)
               END IF
            END IF

            PIX2(WCS(2)) = PIXLAT
            CALL WCSMIX (CTYPE, WCS, WCS(2), 1, LATSPN, 1D0, 0, WORLD,
     *         CRVAL, CEL, PHI, THETA, PRJ, IMG, LIN, PIX2, IERR)
            IF (IERR.NE.0) THEN
               CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
               WRITE (6, '(A,I2)') '  B: WCSMIX error', IERR
            ELSE
               CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA,
     *            PRJ, IMG, LIN, PIX3, IERR)
               IF (IERR.NE.0) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, '(A,I2)') '  B: WCSFWD error', IERR
               ELSE IF (ABS(PIX3(WCS(2))-PIXLAT).GT.TOL .AND.
     *                 (ABS(WORLD(WCS(2))- LAT1).GT.TOL .OR.
     *                  ABS(PIX2(WCS(1))-PIXLNG).GT.TOL)) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, 40) WORLD(WCS(2)), PIX2(WCS(1))
 40               FORMAT ('  B: LAT2 =',F20.15,'    I2 =',F20.15)
               END IF
            END IF

            WORLD(WCS(2)) = LAT1

            PIX2(WCS(1)) = PIXLNG
            CALL WCSMIX (CTYPE, WCS, WCS(1), 2, LNGSPN, 1D0, 0, WORLD,
     *         CRVAL, CEL, PHI, THETA, PRJ, IMG, LIN, PIX2, IERR)
            IF (IERR.NE.0) THEN
               CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
               WRITE (6, '(A,I2)') '  C: WCSMIX error', IERR
            ELSE
               CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA,
     *            PRJ, IMG, LIN, PIX3, IERR)
               IF (IERR.NE.0) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, '(A,I2)') '  C: WCSFWD error', IERR
               ELSE IF (ABS(PIX3(WCS(1))-PIXLNG).GT.TOL .AND.
     *                 (ABS(WORLD(WCS(1))- LNG1).GT.TOL .OR.
     *                  ABS(PIX2(WCS(2))-PIXLAT).GT.TOL)) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, 50) WORLD(WCS(1)), PIX2(WCS(2))
 50               FORMAT ('  C: LNG2 =',F20.15,'    J2 =',F20.15)
               END IF
            END IF

            PIX2(WCS(2)) = PIXLAT
            CALL WCSMIX (CTYPE, WCS, WCS(2), 2, LNGSPN, 1D0, 0, WORLD,
     *         CRVAL, CEL, PHI, THETA, PRJ, IMG, LIN, PIX2, IERR)
            IF (IERR.NE.0) THEN
               CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
               WRITE (6, '(A,I2)') '  D: WCSMIX error', IERR
            ELSE
               CALL WCSFWD (CTYPE, WCS, WORLD, CRVAL, CEL, PHI, THETA,
     *            PRJ, IMG, LIN, PIX3, IERR)
               IF (IERR.NE.0) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, '(A,I2)') '  D: WCSFWD error', IERR
               ELSE IF (ABS(PIX3(WCS(2))-PIXLAT).GT.TOL .AND.
     *                 (ABS(WORLD(WCS(1))- LNG1).GT.TOL .OR.
     *                  ABS(PIX2(WCS(1))-PIXLNG).GT.TOL)) THEN
                  CALL ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
                  WRITE (6, 60) WORLD(WCS(1)), PIX2(WCS(1))
 60               FORMAT ('  D: LNG2 =',F20.15,'    I2 =',F20.15)
               END IF
            END IF

 70      CONTINUE
 80   CONTINUE

      DO 90 K = 0, 20
         PRJ(K) = 0D0
 90   CONTINUE

      RETURN
      END


*-----------------------------------------------------------------------
      SUBROUTINE ID (DOID, PCODE, CEL, LNG1, LAT1, PIXLNG, PIXLAT)
*-----------------------------------------------------------------------
      INTEGER   DOID, IERR
      REAL      IPT(1), JPT(1)
      DOUBLE PRECISION CEL(10), PIXLNG, PIXLAT, LNG1, LAT1, PHI, THETA
      CHARACTER PCODE*3
*-----------------------------------------------------------------------
      IF (DOID.NE.0) THEN
*        Compute native coordinates.
         CALL SPHFWD (LNG1, LAT1, CEL(6), PHI, THETA, IERR)

         WRITE (6, 10) PCODE, LNG1, LAT1, PHI, THETA, PIXLNG,
     *      PIXLAT
 10      FORMAT (A3,': LNG1 =',F20.15,'  LAT1 =',F20.15,/,
     *           '      PHI =',F20.15,' THETA =',F20.15,/,
     *           '       I1 =',F20.15,'    J1 =',F20.15)
         DOID = 0

         CALL PGSCI (9)
         IPT(1) = PIXLNG
         JPT(1) = PIXLAT
         CALL PGPT (1, IPT, JPT, 21)
         CALL PGSCI (2)
      END IF

      RETURN
      END
