*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2000, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   FORTRAN routines for the spherical map projections recognized by the
*   FITS "World Coordinate System" (WCS) convention.
*
*   Summary of routines
*   -------------------
*   Each projection is implemented via separate subroutines for the
*   forward, *FWD, and reverse, *REV, transformation.
*
*   Initialization routines, *SET, compute intermediate values from the
*   projection parameters but need not be called explicitly - see the
*   explanation of PRJ(11) below.
*
*      PRJSET PRJFWD PRJREV   Driver routines (see below).
*
*      AZPSET AZPFWD AZPREV   AZP: zenithal/azimuthal perspective
*      TANSET TANFWD TANREV   TAN: gnomonic
*      STGSET STGFWD STGREV   STG: stereographic
*      SINSET SINFWD SINREV   SIN: orthographic/synthesis
*      ARCSET ARCFWD ARCREV   ARC: zenithal/azimuthal equidistant
*      ZPNSET ZPNFWD ZPNREV   ZPN: zenithal/azimuthal polynomial
*      ZEASET ZEAFWD ZEAREV   ZEA: zenithal/azimuthal equal area
*      AIRSET AIRFWD AIRREV   AIR: Airy
*      CYPSET CYPFWD CYPREV   CYP: cylindrical perspective
*      CEASET CEAFWD CEAREV   CEA: cylindrical equal area
*      CARSET CARFWD CARREV   CAR: Cartesian
*      MERSET MERFWD MERREV   MER: Mercator
*      SFLSET SFLFWD SFLREV   SFL: Sanson-Flamsteed
*      PARSET PARFWD PARREV   PAR: parabolic
*      MOLSET MOLFWD MOLREV   MOL: Molweide
*      AITSET AITFWD AITREV   AIT: Hammer-Aitoff
*      COPSET COPFWD COPREV   COP: conic perspective
*      CODSET CODFWD CODREV   COD: conic equidistant
*      COESET COEFWD COEREV   COE: conic equal area
*      COOSET COOFWD COOREV   COO: conic orthomorphic
*      BONSET BONFWD BONREV   BON: Bonne
*      PCOSET PCOFWD PCOREV   PCO: polyconic
*      TSCSET TSCFWD TSCREV   TSC: tangential spherical cube
*      CSCSET CSCFWD CSCREV   CSC: COBE quadrilateralized spherical cube
*      QSCSET QSCFWD QSCREV   QSC: quadrilateralized spherical cube
*
*
*   Driver routines; PRJSET, PRJFWD & PRJREV
*   ----------------------------------------
*   A set of driver routines are available for use as a generic
*   interface to the specific projection routines.  The interfaces to
*   PRJFWD and PRJREV are the same as those of the forward and reverse
*   transformation routines for the specific projections.  They use the
*   value of PRJ(11) to identify the projection.
*
*   The interface to PRJSET differs from that of the initialization
*   routines for the specific projections and unlike them it must be
*   invoked explicitly to use PRJFWD and PRJREV.
*
*   Given and/or returned:
*      PCODE    C*3      WCS projection code.
*
*   Given and/or returned:
*      PRJ      D(0:20)  Projection parameters (see below).
*
*   Returned:
*      THETA0   D        Native latitude of the reference point, in
*                        degrees.
*
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid projection parameters.
*
*
*   Initialization routine; *SET
*   ----------------------------
*   Initializes elements of a PRJ array which hold intermediate values.
*   Note that this routine need not be called directly; it will be
*   invoked by the forward or reverse projection routine if PRJ(11) is
*   zero.
*
*   Given and/or returned:
*      PRJ      D(0:20)  Projection parameters (see below).
*
*   Returned:
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid projection parameters.
*
*   Forward transformation; *FWD
*   ----------------------------
*   Compute (X,Y) coordinates in the plane of projection from native
*   spherical coordinates (PHI,THETA).
*
*   Given:
*      PHI,     D        Longitude and latitude of the projected point
*      THETA             in native spherical coordinates, in degrees.
*
*   Given and returned:
*      PRJ      D(0:20)  Projection parameters (see below).
*
*   Returned:
*      X,Y      D        Projected coordinates.
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid projection parameters.
*                           2: Invalid value of (phi,theta).
*
*   Reverse transformation; *REV
*   ----------------------------
*   Compute native spherical coordinates (PHI,THETA) from (X,Y)
*   coordinates in the plane of projection.
*
*   Given:
*      X,Y      D        Projected coordinates.
*
*   Given and returned:
*      PRJ      D(0:20)  Projection parameters (see below).
*
*   Returned:
*      PHI,     D        Longitude and latitude of the projected point
*      THETA             in native spherical coordinates, in degrees.
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid projection parameters.
*                           2: Invalid value of (x,y).
*
*   Projection parameters
*   ---------------------
*   The PRJ array consists of the following:
*
*      PRJ(0:9)
*         The first 10 elements contain projection parameters which
*         correspond to the PROJPn keywords in FITS, so PRJ(0) is
*         PROJP0, and PRJ(9) is PROJP9.  Many projections use PRJ(1)
*         PROJP1 and some also use PRJ(2) (PROJP2 ).  ZPN is the only
*         projection which uses any of the others.
*      PRJ(10)
*         r0; The radius of the generating sphere for the projection, a
*         linear scaling parameter.  If this is zero, it will be reset
*         to the default value of 180/pi (the value for FITS WCS).
*      PRJ(11)
*         This must be set to zero whenever any of PRJ(0:10) are set or
*         changed.  This signals the initialization routines to
*         recompute intermediaries.  PRJ(11) may also be set to -1 to
*         disable strict bounds checking for the AZP, TAN, SIN, ZPN, and
*         COP projections.  The initialization routine for each
*         projection resets PRJ(11) to a code unique to the projection.
*
*   The remaining elements of the PRJ array are maintained by the
*   initialization routines and should not be modified.  This is done
*   for the sake of efficiency and to allow an arbitrary number of
*   contexts to be maintained simultaneously.
*
*      PRJ(12:20)
*         Intermediate values derived from the projection parameters.
*
*   Usage of the PRJ array as it applies to each projection is
*   described in the prologue to each trio of projection routines.
*
*   Argument checking
*   -----------------
*   Forward routines:
*
*      The values of phi and theta (the native longitude and latitude)
*      normally lie in the range [-180,180] for phi, and [-90,90] for
*      theta.  However, all forward projections will accept any value
*      of phi and will not normalize it.
*
*      The forward projection routines do not explicitly check that
*      theta lies within the range [-90,90].  They do check for any
*      value of theta which produces an invalid argument to the
*      projection equations (e.g. leading to division by zero).  The
*      forward routines for AZP, TAN, SIN, ZPN, and COP also return
*      error 2 if (phi,theta) corresponds to the overlapped (far) side
*      of the projection but also return the corresponding value of
*      (x,y).  This strict bounds checking may be relaxed by setting
*      PRJ(11) to -1 (rather than 0) when these projections are
*      initialized.
*
*   Reverse routines:
*
*      Error checking on the projected coordinates (x,y) is limited to
*      that required to ascertain whether a solution exists.  Where a
*      solution does exist no check is made that the value of phi and
*      theta obtained lie within the ranges [-180,180] for phi, and
*      [-90,90] for theta.
*
*   Accuracy
*   --------
*   Closure to a precision of at least 1D-10 degree of longitude and
*   latitude has been verified for typical projection parameters on the
*   1 degree grid of native longitude and latitude (to within 5 degrees
*   of any latitude where the projection may diverge).
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: proj.f,v 2.15 2000/12/06 23:53:57 mcalabre Exp $
*=======================================================================
      SUBROUTINE PRJSET (PCODE, PRJ, THETA0, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, IREF(25), NDX
      DOUBLE PRECISION PRJ(0:20), THETA0
      CHARACTER PCODES*99, PCODE*3

      DATA PCODES(01:40) /'AZP TAN STG SIN ARC ZPN ZEA AIR CYP CEA '/
      DATA PCODES(41:80) /'CAR MER SFL PAR MOL AIT COP COE COD COO '/
      DATA PCODES(81:99) /'BON PCO TSC CSC QSC'/
      DATA IREF /2,2,2,2,2,2,2,2,0,0,0,0,0,0,0,0,1,1,1,1,0,0,0,0,0/
*-----------------------------------------------------------------------
*     Determine the projection type.
      NDX = (INDEX(PCODES,PCODE) - 1)/4 + 1
      IF (NDX.EQ.0) THEN
*        Unrecognized projection code.
         IERR = 1
         RETURN
      END IF

*     Native latitude of the reference point.
      IF (IREF(NDX).EQ.0) THEN
         THETA0 = 0D0
      ELSE IF (IREF(NDX).EQ.1) THEN
         THETA0 = PRJ(1)
      ELSE IF (IREF(NDX).EQ.2) THEN
         THETA0 = 90D0
      ELSE
         IERR = 1
         RETURN
      END IF

*     The computed GOTO is used instead of an IF-block for efficiency.
      GO TO (101, 102, 103, 104, 105, 106, 107, 108,
     *       201, 202, 203, 204,
     *       301, 302, 303,
     *       401,
     *       501, 502, 503, 504,
     *       601, 602,
     *       701, 702, 703) NDX

*     AZP: zenithal/azimuthal perspective.
 101  CALL AZPSET (PRJ, IERR)
      RETURN

*     TAN: gnomonic (tan).
 102  CALL TANSET (PRJ, IERR)
      RETURN

*     SIN: orthographic (sine).
 103  CALL STGSET (PRJ, IERR)
      RETURN

*     STG: stereographic.
 104  CALL SINSET (PRJ, IERR)
      RETURN

*     ARC: zenithal/azimuthal equidistant.
 105  CALL ARCSET (PRJ, IERR)
      RETURN

*     ZPN: zenithal/azimuthal polynomial.
 106  CALL ZPNSET (PRJ, IERR)
      RETURN

*     ZEA: zenithal/azimuthal equal area.
 107  CALL ZEASET (PRJ, IERR)
      RETURN

*     AIR: Airy's zenithal projection.
 108  CALL AIRSET (PRJ, IERR)
      RETURN

*     CYP: cylindrical perspective.
 201  CALL CYPSET (PRJ, IERR)
      RETURN

*     CAR: Cartesian.
 202  CALL CEASET (PRJ, IERR)
      RETURN

*     MER: Mercator's.
 203  CALL CARSET (PRJ, IERR)
      RETURN

*     CEA: cylindrical equal area.
 204  CALL MERSET (PRJ, IERR)
      RETURN

*     SFL: Sanson-Flamsteed.
 301  CALL SFLSET (PRJ, IERR)
      RETURN

*     PAR: parabolic.
 302  CALL PARSET (PRJ, IERR)
      RETURN

*     MOL: Mollweide's projection.
 303  CALL MOLSET (PRJ, IERR)
      RETURN

*     AIT: Hammer-Aitoff.
 401  CALL AITSET (PRJ, IERR)
      RETURN

*     COP: conic perspective.
 501  CALL COPSET (PRJ, IERR)
      RETURN

*     COE: conic equal area.
 502  CALL COESET (PRJ, IERR)
      RETURN

*     COD: conic equidistant.
 503  CALL CODSET (PRJ, IERR)
      RETURN

*     COO: conic orthomorphic.
 504  CALL COOSET (PRJ, IERR)
      RETURN

*     BON: Bonne's projection.
 601  CALL BONSET (PRJ, IERR)
      RETURN

*     PCO: polyconic.
 602  CALL PCOSET (PRJ, IERR)
      RETURN

*     TSC: tangential spherical cube.
 701  CALL TSCSET (PRJ, IERR)
      RETURN

*     CSC: COBE quadrilateralized spherical cube.
 702  CALL CSCSET (PRJ, IERR)
      RETURN

*     QSC: quadrilateralized spherical cube.
 703  CALL QSCSET (PRJ, IERR)
      RETURN

      END

*-----------------------------------------------------------------------
      SUBROUTINE PRJFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   I, IERR, J, K, NDX(7,8)
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DATA (NDX(1,J),J=1,8) / 1,  2,  3,  4,  5,  6,  7,  8/
      DATA (NDX(2,J),J=1,8) / 9, 10, 11, 12,  0,  0,  0,  0/
      DATA (NDX(3,J),J=1,8) /13, 14, 15,  0,  0,  0,  0,  0/
      DATA (NDX(4,J),J=1,8) /16,  0,  0,  0,  0,  0,  0,  0/
      DATA (NDX(5,J),J=1,8) /17, 18, 19, 20,  0,  0,  0,  0/
      DATA (NDX(6,J),J=1,8) /21, 22,  0,  0,  0,  0,  0,  0/
      DATA (NDX(7,J),J=1,8) /23, 24, 25,  0,  0,  0,  0,  0/
*-----------------------------------------------------------------------
      K = NINT(ABS(PRJ(11)))
      I = K/100
      J = MOD(K, 100)

      IF (I.LT.1 .OR. I.GT.7 .OR. J.LT.1 .OR. J.GT.8) THEN
         IERR = 1
         RETURN
      END IF


*     The computed GOTO is used instead of an IF-block for efficiency.
      GO TO (101, 102, 103, 104, 105, 106, 107, 108,
     *       201, 202, 203, 204,
     *       301, 302, 303,
     *       401,
     *       501, 502, 503, 504,
     *       601, 602,
     *       701, 702, 703) NDX(I,J)

      IERR = 1
      RETURN

*     AZP: zenithal/azimuthal perspective.
 101  CALL AZPFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     TAN: gnomonic.
 102  CALL TANFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     STG: stereographic.
 103  CALL STGFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     SIN: orthographic.
 104  CALL SINFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     ARC: zenithal/azimuthal equidistant.
 105  CALL ARCFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     ZPN: zenithal/azimuthal polynomial.
 106  CALL ZPNFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     ZEA: zenithal/azimuthal equal area.
 107  CALL ZEAFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     AIR: Airy's zenithal projection.
 108  CALL AIRFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     CYP: cylindrical perspective.
 201  CALL CYPFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     CEA: cylindrical equal area.
 202  CALL CEAFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     CAR: Cartesian.
 203  CALL CARFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     MER: Mercator's.
 204  CALL MERFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     SFL: Sanson-Flamsteed.
 301  CALL SFLFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     PAR: parabolic.
 302  CALL PARFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     MOL: Mollweide's projection.
 303  CALL MOLFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     AIT: Hammer-Aitoff.
 401  CALL AITFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     COP: conic perspective.
 501  CALL COPFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     COE: conic equal area.
 502  CALL COEFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     COD: conic equidistant.
 503  CALL CODFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     COO: conic orthomorphic.
 504  CALL COOFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     BON: Bonne's projection.
 601  CALL BONFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     PCO: polyconic.
 602  CALL PCOFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     TSC: tangential spherical cube.
 701  CALL TSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     CSC: COBE quadrilateralized spherical cube.
 702  CALL CSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

*     QSC: quadrilateralized spherical cube.
 703  CALL QSCFWD (PHI, THETA, PRJ, X, Y, IERR)
      RETURN

      END

*-----------------------------------------------------------------------
      SUBROUTINE PRJREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   I, IERR, J, K, NDX(7,8)
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DATA (NDX(1,J),J=1,8) / 1,  2,  3,  4,  5,  6,  7,  8/
      DATA (NDX(2,J),J=1,8) / 9, 10, 11, 12,  0,  0,  0,  0/
      DATA (NDX(3,J),J=1,8) /13, 14, 15,  0,  0,  0,  0,  0/
      DATA (NDX(4,J),J=1,8) /16,  0,  0,  0,  0,  0,  0,  0/
      DATA (NDX(5,J),J=1,8) /17, 18, 19, 20,  0,  0,  0,  0/
      DATA (NDX(6,J),J=1,8) /21, 22,  0,  0,  0,  0,  0,  0/
      DATA (NDX(7,J),J=1,8) /23, 24, 25,  0,  0,  0,  0,  0/
*-----------------------------------------------------------------------
      K = NINT(ABS(PRJ(11)))
      I = K/100
      J = MOD(K, 100)

      IF (I.LT.1 .OR. I.GT.7 .OR. J.LT.1 .OR. J.GT.8) THEN
         IERR = 1
         RETURN
      END IF


*     The computed GOTO is used instead of an IF-block for efficiency.
      GO TO (101, 102, 103, 104, 105, 106, 107, 108,
     *       201, 202, 203, 204,
     *       301, 302, 303,
     *       401,
     *       501, 502, 503, 504,
     *       601, 602,
     *       701, 702, 703) NDX(I,J)

      IERR = 1
      RETURN

*     AZP: zenithal/azimuthal perspective.
 101  CALL AZPREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     TAN: gnomonic.
 102  CALL TANREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     STG: stereographic.
 103  CALL STGREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     SIN: orthographic.
 104  CALL SINREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     ARC: zenithal/azimuthal equidistant.
 105  CALL ARCREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     ZPN: zenithal/azimuthal polynomial.
 106  CALL ZPNREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     ZEA: zenithal/azimuthal equal area.
 107  CALL ZEAREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     AIR: Airy's zenithal projection.
 108  CALL AIRREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     CYP: cylindrical perspective.
 201  CALL CYPREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     CEA: cylindrical equal area.
 202  CALL CEAREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     CAR: Cartesian.
 203  CALL CARREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     MER: Mercator's.
 204  CALL MERREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     SFL: Sanson-Flamsteed.
 301  CALL SFLREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     PAR: parabolic.
 302  CALL PARREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     MOL: Mollweide's projection.
 303  CALL MOLREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     AIT: Hammer-Aitoff.
 401  CALL AITREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     COP: conic perspective.
 501  CALL COPREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     COE: conic equal area.
 502  CALL COEREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     COD: conic equidistant.
 503  CALL CODREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     COO: conic orthomorphic.
 504  CALL COOREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     BON: Bonne's projection.
 601  CALL BONREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     PCO: polyconic.
 602  CALL PCOREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     TSC: tangential spherical cube.
 701  CALL TSCREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     CSC: COBE quadrilateralized spherical cube.
 702  CALL CSCREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

*     QSC: quadrilateralized spherical cube.
 703  CALL QSCREV (X, Y, PRJ, PHI, THETA, IERR)
      RETURN

      END

*=======================================================================
*   AZP: zenithal/azimuthal perspective projection.
*
*   Given:
*      PRJ(1)    AZP distance parameter, mu in units of r0.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(mu+1)
*      PRJ(13)   1/PRJ(12)
*      PRJ(14)   Boundary parameter, -mu    for |mu| <= 1,
*                                    -1/mu  for |mu| >= 1.
*=======================================================================
      SUBROUTINE AZPSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = PRJ(10)*(PRJ(1) + 1D0)
      IF (PRJ(12).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF
      PRJ(13) = 1D0/PRJ(12)

      IF (ABS(PRJ(1)).LE.1D0) THEN
         PRJ(14) = -PRJ(1)
      ELSE
         PRJ(14) = -1D0/PRJ(1)
      END IF

      IF (PRJ(11).EQ.-1D0) THEN
         PRJ(11) = -101D0
      ELSE
         PRJ(11) =  101D0
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AZPFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, S, STHE, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.101D0) THEN
         CALL AZPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      STHE = SIND(THETA)
      S = PRJ(1) + STHE
      IF (S.EQ.0D0) THEN
         IERR = 2
         RETURN
      END IF

      R =  PRJ(12)*COSD(THETA)/S
      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IF (PRJ(11).GT.0D0 .AND. STHE.LT.PRJ(14)) THEN
         IERR = 2
         RETURN
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AZPREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, RHO, S, THETA, TOL, X, Y

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.101D0) THEN
         CALL AZPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)
      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF

      RHO = R*PRJ(13)
      S = RHO*PRJ(1)/SQRT(RHO*RHO+1D0)
      IF (ABS(S).GT.1D0) THEN
         IF (ABS(S).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         THETA = ATAN2D(1D0,RHO) - SIGN(90D0,S)
      ELSE
         THETA = ATAN2D(1D0,RHO) - ASIND(S)
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   TAN: gnomonic projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*=======================================================================
      SUBROUTINE TANSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      IF (PRJ(11).EQ.-1D0) THEN
         PRJ(11) = -102D0
      ELSE
         PRJ(11) =  102D0
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE TANFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, S, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.102D0) THEN
         CALL TANSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = SIND(THETA)
      IF (S.EQ.0D0) THEN
         IERR = 2
         RETURN
      END IF

      R =  PRJ(10)*COSD(THETA)/S
      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IF (PRJ(11).GT.0D0 .AND. S.LT.0D0) THEN
         IERR = 2
         RETURN
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE TANREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.102D0) THEN
         CALL TANSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)
      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF
      THETA = ATAN2D(PRJ(10), R)

      IERR = 0
      RETURN
      END

*=======================================================================
*   STG: stereographic projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   2*r0
*      PRJ(13)   1/(2*r0)
*=======================================================================
      SUBROUTINE STGSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 360D0/PI
         PRJ(13) = PI/360D0
      ELSE
         PRJ(12) = 2D0*PRJ(10)
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 103D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE STGFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, S, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.103D0) THEN
         CALL STGSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = 1D0 + SIND(THETA)
      IF (S.EQ.0D0) THEN
         IERR = 2
         RETURN
      END IF

      R =  PRJ(12)*COSD(THETA)/S
      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE STGREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D, ATAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.103D0) THEN
         CALL STGSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)
      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF
      THETA = 90D0 - 2D0*ATAND(R*PRJ(13))

      IERR = 0
      RETURN
      END

*=======================================================================
*   SIN: orthographic/synthesis projection.
*
*   Given:
*      PRJ(1:2)  SIN obliqueness parameters, alpha and beta.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   1/r0
*      PRJ(13)   alpha**2 + beta**2
*      PRJ(14)   2*(alpha**2 + beta**2)
*      PRJ(15)   2*(alpha**2 + beta**2 + 1)
*      PRJ(16)   alpha**2 + beta**2 - 1
*=======================================================================
      SUBROUTINE SINSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = 1D0/PRJ(10)
      PRJ(13) = PRJ(1)*PRJ(1) + PRJ(2)*PRJ(2)
      PRJ(14) = 2D0*PRJ(13)
      PRJ(15) = PRJ(14) + 2D0
      PRJ(16) = PRJ(13) - 1D0

      IF (PRJ(11).EQ.-1D0) THEN
         PRJ(11) = -104D0
      ELSE
         PRJ(11) =  104D0
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SINFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION CPHI, CTHE, PHI, PRJ(0:20), SPHI, T, THETA, X, Y,
     *          Z

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION ATAND, COSD, SIND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.104D0) THEN
         CALL SINSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      T = (90D0 - ABS(THETA))*D2R
      IF (T.LT.1D-4) THEN
         IF (THETA.GT.0D0) THEN
            Z = -T*T/2D0
         ELSE
            Z = -2D0 + T*T/2D0
         END IF
         CTHE = T
      ELSE
         Z = SIND(THETA) - 1D0
         CTHE = COSD(THETA)
      END IF

      CPHI = COSD(PHI)
      SPHI = SIND(PHI)
      X =  PRJ(10)*(CTHE*SPHI + PRJ(1)*Z)
      Y = -PRJ(10)*(CTHE*CPHI + PRJ(2)*Z)

*     Validate this solution.
      IF (PRJ(11).GT.0D0) THEN
         IF (PRJ(13).EQ.0D0) THEN
*           Orthographic projection.
            IF (THETA.LT.0D0) THEN
               IERR = 2
               RETURN
            END IF
         ELSE
*           "Synthesis" projection.
            T = ATAND(PRJ(1)*SPHI + PRJ(2)*CPHI)
            IF (THETA.LT.T) THEN
               IERR = 2
               RETURN
            END IF
         END IF
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SINREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, B, C, D, PHI, PRJ(0:20), R2, STH, STH1, STH2,
     *          SXY, THETA, TOL, X, X0, XP, Y, Y0, YP, Z

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION ACOSD, ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.104D0) THEN
         CALL SINSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

*     Compute intermediaries.
      X0 = X*PRJ(12)
      Y0 = Y*PRJ(12)
      R2 = X0*X0 + Y0*Y0

      IF (PRJ(13).EQ.0D0) THEN
*        Orthographic projection.
         IF (R2.NE.0D0) THEN
            PHI = ATAN2D(X0, -Y0)
         ELSE
            PHI = 0D0
         END IF

         IF (R2.LT.0.5D0) THEN
            THETA = ACOSD(SQRT(R2))
         ELSE IF (R2.LE.1D0) THEN
            THETA = ASIND(SQRT(1D0 - R2))
         ELSE
            IERR = 2
            RETURN
         END IF

      ELSE
*        "Synthesis" projection.
         IF (R2.LT.1D-10) THEN
*           Use small angle formula.
            Z = -R2/2D0
            THETA = 90D0 - R2D*SQRT(R2/(1D0 - X0*PRJ(1) + Y0*PRJ(2)))

         ELSE
            SXY = 2D0*(PRJ(1)*X0 - PRJ(2)*Y0)

            A = PRJ(15)
            B = -(SXY + PRJ(14))
            C = R2 + SXY + PRJ(16)
            D = B*B - 2D0*A*C

*           Check for a solution.
            IF (D.LT.0D0) THEN
               IERR = 2
               RETURN
            END IF
            D = SQRT(D)

*           Choose solution closest to pole.
            STH1 = (-B + D)/A
            STH2 = (-B - D)/A
            STH = MAX(STH1,STH2)
            IF (STH.GT.1D0) THEN
               IF (STH-1D0.LT.TOL) THEN
                  STH = 1D0
               ELSE
                  STH = MIN(STH1,STH2)
               END IF
            END IF
            IF (STH.LT.-1D0) THEN
               IF (STH+1D0.GT.-TOL) STH = -1D0
            END IF
            IF (STH.GT.1D0 .OR. STH.LT.-1D0) THEN
               IERR = 2
               RETURN
            END IF

            THETA = ASIND(STH)
            Z = STH - 1D0
         END IF

         XP = -Y0 - PRJ(2)*Z
         YP =  X0 - PRJ(1)*Z
         IF (XP.EQ.0D0 .AND. YP.EQ.0D0) THEN
            PHI = 0D0
         ELSE
            PHI = ATAN2D(YP,XP)
         END IF
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   ARC: zenithal/azimuthal equidistant projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*=======================================================================
      SUBROUTINE ARCSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 105D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ARCFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.105D0) THEN
         CALL ARCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R =  PRJ(12)*(90D0 - THETA)
      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ARCREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.105D0) THEN
         CALL ARCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)
      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF
      THETA = 90D0 - R*PRJ(13)

      IERR = 0
      RETURN
      END

*=======================================================================
*   ZPN: zenithal/azimuthal polynomial projection.
*
*   Given:
*      PRJ(0:9)  Polynomial coefficients.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   Degree of the polynomial, N.
*      PRJ(13)   Co-latitude of the first point of inflection (N > 2).
*      PRJ(14)   Radius of the first point of inflection (N > 2).
*=======================================================================
      SUBROUTINE ZPNSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, I, J, K
      DOUBLE PRECISION D, D1, D2, PRJ(0:20), R, TOL, ZD, ZD1, ZD2

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-13)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

*     Find the highest non-zero coefficient.
      DO 10 K = 9, 0, -1
         IF (PRJ(K).NE.0D0) GO TO 20
 10   CONTINUE
      IERR = 1
      RETURN

 20   PRJ(12) = DBLE(K)

      IF (K.GE.3) THEN
*        Find the point of inflection closest to the pole.
         ZD1 = 0D0
         D1  = PRJ(1)
         IF (D1.LE.0D0) THEN
            IERR = 1
            RETURN
         END IF

*        Find the point where the derivative first goes negative.
         DO 40 I = 1, 180
            ZD2 = PI*DBLE(I)/180D0
            D2  = 0D0
            DO 30 J = K, 1, -1
               D2 = D2*ZD2 + J*PRJ(J)
 30         CONTINUE

            IF (D2.LE.0D0) GO TO 50
            ZD1 = ZD2
            D1  = D2
 40      CONTINUE

*        No negative derivative -> no point of inflection.
         ZD = PI
         GO TO 80

*        Find where the derivative is zero.
 50      DO 70 I = 1, 10
            ZD = ZD1 - D1*(ZD2-ZD1)/(D2-D1)

            D = 0D0
            DO 60 J = K, 1, -1
               D = D*ZD + J*PRJ(J)
 60         CONTINUE

            IF (ABS(D).LT.TOL) GO TO 80

            IF (D.LT.0D0) THEN
               ZD2 = ZD
               D2  = D
            ELSE
               ZD1 = ZD
               D1  = D
            END IF
 70      CONTINUE

 80      R = 0D0
         DO 90 J = K, 0, -1
            R = R*ZD + PRJ(J)
 90      CONTINUE
         PRJ(13) = ZD
         PRJ(14) = R
      END IF

      IF (PRJ(11).EQ.-1D0) THEN
         PRJ(11) = -106D0
      ELSE
         PRJ(11) =  106D0
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ZPNFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J
      DOUBLE PRECISION PHI, PRJ(0:20), R, S, THETA, X, Y

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.106D0) THEN
         CALL ZPNSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = (90D0 - THETA)*D2R

      R = 0D0
      DO 10 J = 9, 0, -1
         R = R*S + PRJ(J)
 10   CONTINUE
      R = PRJ(10)*R

      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IF (PRJ(11).GT.0D0 .AND. S.GT.PRJ(13)) THEN
         IERR = 2
         RETURN
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ZPNREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, I, J, K
      DOUBLE PRECISION A, B, C, D, LAMBDA, PHI, PRJ(0:20), R, R1, R2,
     *          RT, THETA, TOL, X, Y, ZD, ZD1, ZD2

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION ATAN2D
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.106D0) THEN
         CALL ZPNSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      K = NINT(PRJ(12))

      R = SQRT(X*X + Y*Y)/PRJ(10)

      IF (K.LT.1) THEN
*        Constant - no solution.
         IERR = 1
         RETURN
      ELSE IF (K.EQ.1) THEN
*        Linear.
         ZD = (R - PRJ(0))/PRJ(1)
      ELSE IF (K.EQ.2) THEN
*        Quadratic.
         A = PRJ(2)
         B = PRJ(1)
         C = PRJ(0) - R

         D = B*B - 4.0*A*C
         IF (D.LT.0D0) THEN
            IERR = 2
            RETURN
         END IF
         D = SQRT(D)

*        Choose solution closest to pole.
         ZD1 = (-B + D)/(2D0*A)
         ZD2 = (-B - D)/(2D0*A)
         ZD  = MIN(ZD1,ZD2)
         IF (ZD.LT.-TOL) ZD = MAX(ZD1,ZD2)
         IF (ZD.LT.0D0) THEN
            IF (ZD.LT.-TOL) THEN
               IERR = 2
               RETURN
            END IF
            ZD = 0D0
         ELSE IF (ZD.GT.PI) THEN
            IF (ZD.GT.PI+TOL) THEN
               IERR = 2
               RETURN
            END IF
            ZD = PI
         END IF
      ELSE
*        Higher order - solve iteratively.
         ZD1 = 0D0
         R1  = PRJ(0)
         ZD2 = PRJ(13)
         R2  = PRJ(14)

         IF (R.LT.R1) THEN
            IF (R.LT.R1-TOL) THEN
               IERR = 2
               RETURN
            END IF
            ZD = ZD1
            GO TO 120
         ELSE IF (R.GT.R2) THEN
            IF (R.GT.R2+TOL) THEN
               IERR = 2
               RETURN
            END IF
            ZD = ZD2
            GO TO 120
         ELSE
*           Disect the interval.
            DO 110 J = 1, 100
               LAMBDA = (R2 - R)/(R2 - R1)
               IF (LAMBDA.LT.0.1D0) THEN
                  LAMBDA = 0.1D0
               ELSE IF (LAMBDA.GT.0.9D0) THEN
                  LAMBDA = 0.9D0
               END IF

               ZD = ZD2 - LAMBDA*(ZD2 - ZD1)

               RT = 0D0
               DO 100 I = K, 0, -1
                   RT = (RT * ZD) + PRJ(I)
 100           CONTINUE

               IF (RT.LT.R) THEN
                   IF (R-RT.LT.TOL) GO TO 120
                   R1 = RT
                   ZD1 = ZD
               ELSE
                   IF (RT-R.LT.TOL) GO TO 120
                   R2 = RT
                   ZD2 = ZD
               END IF

               IF (ABS(ZD2-ZD1).LT.TOL) GO TO 120
 110        CONTINUE
         END IF
      END IF

 120  IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF
      THETA = 90D0 - ZD*R2D

      IERR = 0
      RETURN
      END

*=======================================================================
*   ZEA: zenithal/azimuthal equal area projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   2*r0
*      PRJ(13)   1/(2*r0)
*=======================================================================
      SUBROUTINE ZEASET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 360D0/PI
         PRJ(13) = PI/360D0
      ELSE
         PRJ(12) = 2D0*PRJ(10)
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 107D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ZEAFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.107D0) THEN
         CALL ZEASET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R =  PRJ(12)*SIND((90D0 - THETA)/2D0)
      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE ZEAREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), R, S, THETA, TOL, X, Y

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.107D0) THEN
         CALL ZEASET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)
      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF

      S = R*PRJ(13)
      IF (ABS(S).GT.1D0) THEN
         IF (ABS(R-PRJ(12)).LT.TOL) THEN
            THETA = -90D0
         ELSE
            IERR = 2
            RETURN
         END IF
      ELSE
         THETA = 90D0 - 2D0*ASIND(R*PRJ(13))
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   AIR: Airy's projection.
*
*   Given:
*      PRJ(1)    Latitude theta_b within which the error is minimized,
*                in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   2*r0
*      PRJ(13)   ln(cos(xi_b))/tan(xi_b)**2, where xi_b = (90-theta_b)/2
*      PRJ(14)   1/2 - PRJ(13)
*      PRJ(15)   2*r0*PRJ(14)
*      PRJ(16)   tol, cutoff for using small angle approximation, in
*                radians.
*      PRJ(17)   PRJ(14)*tol
*      PRJ(18)   (180/pi)/PRJ(14)
*=======================================================================
      SUBROUTINE AIRSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION CXI, PRJ(0:20), TOL

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-4)

      DOUBLE PRECISION COSD
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = 2D0*PRJ(10)
      IF (PRJ(1).EQ.90D0) THEN
         PRJ(13) = -0.5D0
         PRJ(14) = 1D0
      ELSE IF (PRJ(1).GT.-90D0) THEN
         CXI = COSD((90D0 - PRJ(1))/2D0)
         PRJ(13) = LOG(CXI)*(CXI*CXI)/(1D0-CXI*CXI)
         PRJ(14) = 0.5D0 - PRJ(13)
      ELSE
         IERR = 1
         RETURN
      END IF

      PRJ(15) = PRJ(12)*PRJ(14)
      PRJ(16) = TOL
      PRJ(17) = PRJ(14)*TOL
      PRJ(18) = R2D/PRJ(14)

      PRJ(11) = 108D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AIRFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION CXI, PHI, PRJ(0:20), R, THETA, TXI, X, XI, Y

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.108D0) THEN
         CALL AIRSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      IF (THETA.EQ.90D0) THEN
         R = 0D0
      ELSE IF (THETA.GT.-90D0) THEN
         XI = D2R*(90D0 - THETA)/2D0
         IF (XI.LT.PRJ(16)) THEN
            R = XI*PRJ(15)
         ELSE
            CXI = COS(XI)
            TXI = SQRT(1D0-CXI*CXI)/CXI
            R = -PRJ(12)*(LOG(CXI)/TXI + PRJ(13)*TXI)
         END IF
      ELSE
         IERR = 2
         RETURN
      END IF

      X =  R*SIND(PHI)
      Y = -R*COSD(PHI)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AIRREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J
      DOUBLE PRECISION CXI, LAMBDA, PHI, PRJ(0:20), R, R1, R2, RT,
     *          THETA, TOL, TXI, X, X1, X2, XI, Y

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ACOSD, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.108D0) THEN
         CALL AIRSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = SQRT(X*X + Y*Y)/PRJ(12)

      IF (R.EQ.0D0) THEN
         XI = 0D0
      ELSE IF (R.LT.PRJ(17)) THEN
         XI = R*PRJ(18)
      ELSE
*        Find a solution interval.
         X1 = 1D0
         R1 = 0D0
         DO 10 J = 1, 30
            X2 = X1/2D0
            TXI = SQRT(1D0-X2*X2)/X2
            R2 = -(LOG(X2)/TXI + PRJ(13)*TXI)

            IF (R2.GE.R) GO TO 20
            X1 = X2
            R1 = R2
 10      CONTINUE
         IERR = 2
         RETURN

 20      DO 30 J = 1, 100
*           Weighted division of the interval.
            LAMBDA = (R2-R)/(R2-R1)
            IF (LAMBDA.LT.0.1D0) THEN
               LAMBDA = 0.1D0
            ELSE IF (LAMBDA.GT.0.9D0) THEN
               LAMBDA = 0.9D0
            END IF
            CXI = X2 - LAMBDA*(X2-X1)

            TXI = SQRT(1D0-CXI*CXI)/CXI
            RT = -(LOG(CXI)/TXI + PRJ(13)*TXI)

            IF (RT.LT.R) THEN
               IF (R-RT.LT.TOL) GO TO 40
               R1 = RT
               X1 = CXI
            ELSE
               IF (RT-R.LT.TOL) GO TO 40
               R2 = RT
               X2 = CXI
            END IF
 30      CONTINUE
         IERR = 2
         RETURN

 40      XI = ACOSD(CXI)
      END IF

      IF (R.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(X, -Y)
      END IF
      THETA = 90D0 - 2D0*XI

      IERR = 0
      RETURN
      END

*=======================================================================
*   CYP: cylindrical perspective projection.
*
*   Given:
*      PRJ(1)    Distance of point of projection from the centre of the
*                generating sphere, mu, in units of r0.
*      PRJ(2)    Radius of the cylinder of projection, lambda, in units
*                of r0.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*lambda*(pi/180)
*      PRJ(13)   (180/pi)/(r0*lambda)
*      PRJ(14)   r0*(mu + lambda)
*      PRJ(15)   1/(r0*(mu + lambda))
*=======================================================================
      SUBROUTINE CYPSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D

         PRJ(12) = PRJ(2)
         IF (PRJ(12).EQ.0D0) THEN
            IERR = 1
            RETURN
         END IF

         PRJ(13) = 1D0/PRJ(12)

         PRJ(14) = R2D*(PRJ(1) + PRJ(2))
         IF (PRJ(14).EQ.0D0) THEN
            IERR = 1
            RETURN
         END IF

         PRJ(15) = 1D0/PRJ(14)
      ELSE
         PRJ(12) = PRJ(10)*PRJ(2)*D2R
         IF (PRJ(12).EQ.0D0) THEN
            IERR = 1
            RETURN
         END IF

         PRJ(13) = 1D0/PRJ(12)

         PRJ(14) = PRJ(10)*(PRJ(1) + PRJ(2))
         IF (PRJ(14).EQ.0D0) THEN
            IERR = 1
            RETURN
         END IF

         PRJ(15) = 1D0/PRJ(14)
      END IF

      PRJ(11) = 201D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CYPFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.201D0) THEN
         CALL CYPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = PRJ(1) + COSD(THETA)
      IF (S.EQ.0D0) THEN
            IERR = 2
            RETURN
         END IF

      X = PRJ(12)*PHI
      Y = PRJ(14)*SIND(THETA)/S

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CYPREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION ETA, PHI, PRJ(0:20), THETA, X, Y

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.201D0) THEN
         CALL CYPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      PHI   = X*PRJ(13)
      ETA   = Y*PRJ(15)
      THETA = ATAN2D(ETA,1D0) + ASIND(ETA*PRJ(1)/SQRT(ETA*ETA+1D0))

      IERR = 0
      RETURN
      END

*=======================================================================
*   CEA: cylindrical equal area projection.
*
*   Given:
*      PRJ(1)    Square of the cosine of the latitude at which the
*                projection is conformal, lambda.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*      PRJ(14)   r0/lambda
*      PRJ(15)   lambda/r0
*=======================================================================
      SUBROUTINE CEASET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
         IF (PRJ(1).LE.0D0 .OR. PRJ(1).GT.1D0) THEN
            IERR = 1
            RETURN
         END IF
         PRJ(14) = PRJ(10)/PRJ(1)
         PRJ(15) = PRJ(1)/PRJ(10)
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = R2D/PRJ(10)
         IF (PRJ(1).LE.0D0 .OR. PRJ(1).GT.1D0) THEN
            IERR = 1
            RETURN
         END IF
         PRJ(14) = PRJ(10)/PRJ(1)
         PRJ(15) = PRJ(1)/PRJ(10)
      END IF

      PRJ(11) = 202D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CEAFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DOUBLE PRECISION SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.202D0) THEN
         CALL CEASET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      X = PRJ(12)*PHI
      Y = PRJ(14)*SIND(THETA)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CEAREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, THETA, TOL, X, Y

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION ASIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.202D0) THEN
         CALL CEASET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = Y*PRJ(15)
      IF (ABS(S).GT.1D0) THEN
         IF (ABS(S).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF

         S = SIGN(1D0,S)
      END IF

      PHI   = X*PRJ(13)
      THETA = ASIND(S)

      IERR = 0
      RETURN
      END

*=======================================================================
*   CAR: Cartesian projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*=======================================================================
      SUBROUTINE CARSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 203D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CARFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.203D0) THEN
         CALL CARSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      X = PRJ(12)*PHI
      Y = PRJ(12)*THETA

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CARREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.203D0) THEN
         CALL CARSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      PHI   = PRJ(13)*X
      THETA = PRJ(13)*Y

      IERR = 0
      RETURN
      END

*=======================================================================
*   MER: Mercator's projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*=======================================================================
      SUBROUTINE MERSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 204D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE MERFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DOUBLE PRECISION TAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.204D0) THEN
         CALL MERSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      IF (THETA.LE.-90D0 .OR. THETA.GE.90D0) THEN
         IERR = 2
         RETURN
      END IF

      X = PRJ(12)*PHI
      Y = PRJ(10)*LOG(TAND((90D0+THETA)/2D0))

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE MERREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DOUBLE PRECISION ATAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.204D0) THEN
         CALL MERSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      PHI   = X*PRJ(13)
      THETA = 2D0*ATAND(EXP(Y/PRJ(10))) - 90D0

      IERR = 0
      RETURN
      END

*=======================================================================
*   SFL: Sanson-Flamsteed ("global sinusoid") projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*=======================================================================
      SUBROUTINE SFLSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 301D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SFLFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, X, Y

      DOUBLE PRECISION COSD
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.301D0) THEN
         CALL SFLSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      X = PRJ(12)*PHI*COSD(THETA)
      Y = PRJ(12)*THETA

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SFLREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), THETA, W, X, Y
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.301D0) THEN
         CALL SFLSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      W = COS(Y/PRJ(10))
      IF (W.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = X*PRJ(13)/COS(Y/PRJ(10))
      END IF
      THETA = Y*PRJ(13)

      IERR = 0
      RETURN
      END

*=======================================================================
*   PAR: parabolic projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   (180/pi)/r0
*      PRJ(14)   pi*r0
*      PRJ(15)   1/(pi*r0)
*=======================================================================
      SUBROUTINE PARSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
         PRJ(14) = 180D0
         PRJ(15) = 1D0/PRJ(14)
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
         PRJ(14) = PI*PRJ(10)
         PRJ(15) = 1D0/PRJ(14)
      END IF

      PRJ(11) = 302D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE PARFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, THETA, X, Y

      DOUBLE PRECISION SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.302D0) THEN
         CALL PARSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = SIND(THETA/3D0)
      X = PRJ(12)*PHI*(1D0 - 4D0*S*S)
      Y = PRJ(14)*S

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE PARREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, T, THETA, X, Y

      DOUBLE PRECISION ASIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.302D0) THEN
         CALL PARSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      S = Y*PRJ(15)
      IF (S.GT.1D0 .OR. S.LT.-1D0) THEN
         IERR = 2
         RETURN
      END IF

      T = 1D0 - 4D0*S*S
      IF (T.EQ.0D0) THEN
         IF (X.EQ.0D0) THEN
            PHI = 0D0
         ELSE
            IERR = 2
            RETURN
         END IF
      ELSE
         PHI = PRJ(13)*X/T
      END IF

      THETA = 3D0*ASIND(S)

      IERR = 0
      RETURN
      END

*=======================================================================
*   MOL: Mollweide's projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   sqrt(2)*r0
*      PRJ(13)   sqrt(2)*r0/90
*      PRJ(14)   1/(sqrt(2)*r0)
*      PRJ(15)   90/r0
*      PRJ(16)   2/pi
*=======================================================================
      SUBROUTINE MOLSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION SQRT2
      PARAMETER (SQRT2 = 1.4142135623730950488D0)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = SQRT2*PRJ(10)
      PRJ(13) = PRJ(12)/90D0
      PRJ(14) = 1D0/PRJ(12)
      PRJ(15) = 90D0/PRJ(10)
      PRJ(16) = 2D0/PI

      PRJ(11) = 303D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE MOLFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J
      DOUBLE PRECISION ALPHA, PHI, PRJ(0:20), RESID, THETA, TOL, U,
     *          V, V0, V1, X, Y

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.303D0) THEN
         CALL MOLSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      IF (ABS(THETA).EQ.90D0) THEN
         X = 0D0
         Y = SIGN(PRJ(12),THETA)
      ELSE IF (THETA.EQ.0D0) THEN
         X = PRJ(13)*PHI
         Y = 0D0
      ELSE
         U  = PI*SIND(THETA)
         V0 = -PI
         V1 = +PI
         V  = U
         DO 10 J = 1, 100
            RESID = (V - U) + SIN(V)
            IF (RESID.LT.0D0) THEN
               IF (RESID.GT.-TOL) GO TO 20
               V0 = V
            ELSE
               IF (RESID.LT.TOL) GO TO 20
               V1 = V
            END IF
            V = (V0 + V1)/2D0
 10      CONTINUE

 20      ALPHA = V/2D0
         X = PRJ(13)*PHI*COS(ALPHA)
         Y = PRJ(12)*SIN(ALPHA)
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE MOLREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, THETA, TOL, X, Y, Y0, Z

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ASIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.303D0) THEN
         CALL MOLSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      Y0 = Y/PRJ(10)
      S  = 2D0 - Y0*Y0
      IF (S.LE.TOL) THEN
         IF (S.LT.-TOL) THEN
            IERR = 2
            RETURN
         END IF
         S = 0D0

         IF (ABS(X).GT.TOL) THEN
            IERR = 2
            RETURN
         END IF
         PHI = 0D0
      ELSE
         S = SQRT(S)
         PHI = PRJ(15)*X/S
      END IF

      Z = Y*PRJ(14)
      IF (ABS(Z).GT.1D0) THEN
         IF (ABS(Z).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         Z = SIGN(1D0,Z) + Y0*S/PI
      ELSE
         Z = ASIN(Z)*PRJ(16) + Y0*S/PI
      END IF

      IF (ABS(Z).GT.1D0) THEN
         IF (ABS(Z).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         Z = SIGN(1D0,Z)
      END IF

      THETA = ASIND(Z)

      IERR = 0
      RETURN
      END

*=======================================================================
*   AIT: Hammer-Aitoff projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   2*r0**2
*      PRJ(13)   1/(2*r0)**2
*      PRJ(14)   1/(4*r0)**2
*      PRJ(15)   1/(2*r0)
*=======================================================================
      SUBROUTINE AITSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = 2D0*PRJ(10)*PRJ(10)
      PRJ(13) = 1D0/(2D0*PRJ(12))
      PRJ(14) = PRJ(13)/4D0
      PRJ(15) = 1D0/(2D0*PRJ(10))

      PRJ(11) = 401D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AITFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COSTHE, PHI, PRJ(0:20), THETA, W, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.401D0) THEN
         CALL AITSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      COSTHE = COSD(THETA)
      W = SQRT(PRJ(12)/(1D0 + COSTHE*COSD(PHI/2D0)))
      X = 2D0*W*COSTHE*SIND(PHI/2D0)
      Y = W*SIND(THETA)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE AITREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PHI, PRJ(0:20), S, THETA, TOL, U, X, XP, Y, YP, Z

      PARAMETER (TOL = 1D-13)

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.401D0) THEN
         CALL AITSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      U = 1D0 - X*X*PRJ(14) - Y*Y*PRJ(13)
      IF (U.LT.0D0) THEN
         IF (U.LT.-TOL) THEN
            IERR = 2
            RETURN
         END IF

         U = 0D0
      END IF

      Z = SQRT(U)
      S = Z*Y/PRJ(10)
      IF (ABS(S).GT.1D0) THEN
         IF (ABS(S).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF

         S = SIGN(1D0,S)
      END IF

      XP = 2D0*Z*Z - 1D0
      YP = Z*X*PRJ(15)
      IF (XP.EQ.0D0 .AND. YP.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = 2D0*ATAN2D(YP, XP)
      END IF
      THETA = ASIND(S)

      IERR = 0
      RETURN
      END

*=======================================================================
*   COP: conic perspective projection.
*
*   Given:
*      PRJ(1)    sigma = (theta2+theta1)/2
*      PRJ(2)    delta = (theta2-theta1)/2, where theta1 and theta2 are
*                the latitudes of the standard parallels, in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   C  = sin(sigma)
*      PRJ(13)   1/C
*      PRJ(14)   Y0 = r0*cos(delta)*cot(sigma)
*      PRJ(15)   r0*cos(delta)
*      PRJ(16)   1/(r0*cos(delta))
*      PRJ(17)   cot(sigma)
*=======================================================================
      SUBROUTINE COPSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND, TAND
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      PRJ(12) = SIND(PRJ(1))
      IF (PRJ(12).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF

      PRJ(13) = 1D0/PRJ(12)

      PRJ(15) = PRJ(10)*COSD(PRJ(2))
      IF (PRJ(15).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF

      PRJ(16) = 1D0/PRJ(15)
      PRJ(17) = 1D0/TAND(PRJ(1))

      PRJ(14) = PRJ(15)*PRJ(17)

      IF (PRJ(11).EQ.-1D0) THEN
         PRJ(11) = -501D0
      ELSE
         PRJ(11) =  501D0
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COPFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, PHI, PRJ(0:20), R, S, T, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.501D0) THEN
         CALL COPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      T = THETA - PRJ(1)
      S = COSD(T)
      IF (S.EQ.0D0) THEN
         IERR = 2
         RETURN
      END IF

      A = PRJ(12)*PHI
      R = PRJ(14) - PRJ(15)*SIND(T)/S

      X =           R*SIND(A)
      Y = PRJ(14) - R*COSD(A)

      IF (PRJ(11).GT.0D0 .AND. R*PRJ(12).LT.0D0) THEN
         IERR = 2
         RETURN
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COPREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, DY, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D, ATAND
*-----------------------------------------------------------------------
      IF (ABS(PRJ(11)).NE.501D0) THEN
         CALL COPSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      DY = PRJ(14) - Y
      R  = SQRT(X*X + DY*DY)
      IF (PRJ(1).LT.0D0) R = -R

      IF (R.EQ.0D0) THEN
         A = 0D0
      ELSE
         A = ATAN2D(X/R, DY/R)
      END IF

      PHI   = A*PRJ(13)
      THETA = PRJ(1) + ATAND(PRJ(17) - R*PRJ(16))

      IERR = 0
      RETURN
      END

*=======================================================================
*   COE: conic equal area projection.
*
*   Given:
*      PRJ(1)    sigma = (theta2+theta1)/2
*      PRJ(2)    delta = (theta2-theta1)/2, where theta1 and theta2 are
*                the latitudes of the standard parallels, in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   C = (sin(theta1) + sin(theta2))/2
*      PRJ(13)   1/C
*      PRJ(14)   Y0 = chi*sqrt(psi - 2C*sind(sigma))
*      PRJ(15)   chi = r0/C
*      PRJ(16)   psi = 1 + sin(theta1)*sin(theta2)
*      PRJ(17)   2C
*      PRJ(18)   (1 + sin(theta1)*sin(theta2))*(r0/C)**2
*      PRJ(19)   C/(2*r0**2)
*      PRJ(20)   Y0 = chi*sqrt(psi + 2C)
*=======================================================================
      SUBROUTINE COESET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20), THETA1, THETA2

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION SIND
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      THETA1 = PRJ(1) - PRJ(2)
      THETA2 = PRJ(1) + PRJ(2)

      PRJ(12) = (SIND(THETA1) + SIND(THETA2))/2D0
      IF (PRJ(12).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF

      PRJ(13) = 1D0/PRJ(12)

      PRJ(15) = PRJ(10)/PRJ(12)
      PRJ(16) = 1D0 + SIND(THETA1)*SIND(THETA2)
      PRJ(17) = 2D0*PRJ(12)
      PRJ(18) = PRJ(15)*PRJ(15)*PRJ(16)
      PRJ(19) = 1D0/(2D0*PRJ(10)*PRJ(15))
      PRJ(20) = PRJ(15)*SQRT(PRJ(16) + PRJ(17))

      PRJ(14) = PRJ(15)*SQRT(PRJ(16) - PRJ(17)*SIND(PRJ(1)))

      PRJ(11) = 502D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COEFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.502D0) THEN
         CALL COESET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      A = PHI*PRJ(12)
      IF (THETA.EQ.-90D0) THEN
         R = PRJ(20)
      ELSE
         R = PRJ(15)*SQRT(PRJ(16) - PRJ(17)*SIND(THETA))
      END IF

      X =           R*SIND(A)
      Y = PRJ(14) - R*COSD(A)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COEREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, DY, PHI, PRJ(0:20), R, THETA, TOL, W, X, Y

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.502D0) THEN
         CALL COESET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      DY = PRJ(14) - Y
      R  = SQRT(X*X + DY*DY)
      IF (PRJ(1).LT.0D0) R = -R

      IF (R.EQ.0D0) THEN
         A = 0D0
      ELSE
         A = ATAN2D(X/R, DY/R)
      END IF

      PHI = A*PRJ(13)
      IF (ABS(R - PRJ(20)).LT.TOL) THEN
         THETA = -90D0
      ELSE
         W = (PRJ(18) - R*R)*PRJ(19)
         IF (ABS(W).GT.1D0) THEN
            IF (ABS(W-1D0).LT.TOL) THEN
               THETA = 90D0
            ELSE IF (ABS(W+1D0).LT.TOL) THEN
               THETA = -90D0
            ELSE
               IERR = 2
               RETURN
            END IF
         ELSE
            THETA = ASIND(W)
         END IF
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   COD: conic equidistant projection.
*
*   Given:
*      PRJ(1)    sigma = (theta2+theta1)/2
*      PRJ(2)    delta = (theta2-theta1)/2, where theta1 and theta2 are
*                the latitudes of the standard parallels, in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   C = r0*sin(sigma)*sin(delta)/delta
*      PRJ(13)   1/C
*      PRJ(14)   Y0 = delta*cot(delta)*cot(sigma)
*      PRJ(15)   Y0 + sigma
*=======================================================================
      SUBROUTINE CODSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      IF (PRJ(2).EQ.0D0) THEN
         PRJ(12) = PRJ(10)*SIND(PRJ(1))*D2R
      ELSE
         PRJ(12) = PRJ(10)*SIND(PRJ(1))*SIND(PRJ(2))/PRJ(2)
      END IF

      IF (PRJ(12).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF

      PRJ(13) = 1D0/PRJ(12)
      PRJ(14) = PRJ(10)*COSD(PRJ(2))*COSD(PRJ(1))/PRJ(12)
      PRJ(15) = PRJ(14) + PRJ(1)

      PRJ(11) = 503D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CODFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.503D0) THEN
         CALL CODSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      A = PRJ(12)*PHI
      R = PRJ(15) - THETA

      X =           R*SIND(A)
      Y = PRJ(14) - R*COSD(A)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CODREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, DY, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.503D0) THEN
         CALL CODSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      DY = PRJ(14) - Y
      R  = SQRT(X*X + DY*DY)
      IF (PRJ(1).LT.0D0) R = -R

      IF (R.EQ.0D0) THEN
         A = 0D0
      ELSE
         A = ATAN2D(X/R, DY/R)
      END IF

      PHI   = A*PRJ(13)
      THETA = PRJ(15) - R

      IERR = 0
      RETURN
      END

*=======================================================================
*   COO: conic orthomorphic projection.
*
*   Given:
*      PRJ(1)    sigma = (theta2+theta1)/2
*      PRJ(2)    delta = (theta2-theta1)/2, where theta1 and theta2 are
*                the latitudes of the standard parallels, in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   C = ln(cos(theta2)/cos(theta1))/ln(tan(tau2)/tan(tau1))
*                where tau1 = (90 - theta1)/2
*                      tau2 = (90 - theta2)/2
*      PRJ(13)   1/C
*      PRJ(14)   Y0 = psi*tan((90-sigma)/2)**C
*      PRJ(15)   psi = (r0*cos(theta1)/C)/tan(tau1)**C
*      PRJ(16)   1/psi
*=======================================================================
      SUBROUTINE COOSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COS1, COS2, PRJ(0:20), TAN1, TAN2, THETA1, THETA2

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND, TAND
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) PRJ(10) = R2D

      THETA1 = PRJ(1) - PRJ(2)
      THETA2 = PRJ(1) + PRJ(2)

      TAN1 = TAND((90D0 - THETA1)/2D0)
      COS1 = COSD(THETA1)

      IF (THETA1.EQ.THETA2) THEN
         PRJ(12) = SIND(THETA1)
      ELSE
         TAN2 = TAND((90D0 - THETA2)/2D0)
         COS2 = COSD(THETA2)
         PRJ(12) = LOG(COS2/COS1)/LOG(TAN2/TAN1)
      END IF
      IF (PRJ(12).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF

      PRJ(13) = 1D0/PRJ(12)

      PRJ(15) = PRJ(10)*(COS1/PRJ(12))/(TAN1)**PRJ(12)
      IF (PRJ(15).EQ.0D0) THEN
         IERR = 1
         RETURN
      END IF
      PRJ(16) = 1D0/PRJ(15)
      PRJ(14) = PRJ(15)*TAND((90D0-PRJ(1))/2D0)**PRJ(12)

      PRJ(11) = 504D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COOFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND, TAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.504D0) THEN
         CALL COOSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      A = PRJ(12)*PHI
      IF (THETA.EQ.-90D0) THEN
         IF (PRJ(12).LT.0D0) THEN
            R = 0D0
         ELSE
            IERR = 2
            RETURN
         END IF
      ELSE
         R = PRJ(15)*TAND((90D0 - THETA)/2D0)**PRJ(12)
      END IF

      X =           R*SIND(A)
      Y = PRJ(14) - R*COSD(A)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE COOREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, DY, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D, ATAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.504D0) THEN
         CALL COOSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      DY = PRJ(14) - Y
      R  = SQRT(X*X + DY*DY)
      IF (PRJ(1).LT.0D0) R = -R

      IF (R.EQ.0D0) THEN
         A = 0D0
      ELSE
         A = ATAN2D(X/R, DY/R)
      END IF

      PHI = A*PRJ(13)
      IF (R.EQ.0D0) THEN
         IF (PRJ(12).LT.0D0) THEN
            THETA = -90D0
         ELSE
            RETURN
         END IF
      ELSE
         THETA = 90D0 - 2D0*ATAND((R*PRJ(16))**PRJ(13))
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   BON: Bonne's projection.
*
*   Given:
*      PRJ(1)    Bonne conformal latitude, theta1, in degrees.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*pi/180
*      PRJ(13)   Y0 = r0*(cot(theta1) + theta1*pi/180)
*=======================================================================
      SUBROUTINE BONSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = PRJ(10)*COSD(PRJ(1))/SIND(PRJ(1)) + PRJ(1)
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = PRJ(10)*(COSD(PRJ(1))/SIND(PRJ(1)) + PRJ(1)*D2R)
      END IF

      PRJ(11) = 601D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE BONFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(1).EQ.0D0) THEN
*        Sanson-Flamsteed.
         CALL SFLFWD (PHI, THETA, PRJ, X, Y, IERR)
         RETURN
      END IF

      IF (PRJ(11).NE.601D0) THEN
         CALL BONSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      R = PRJ(13) - THETA*PRJ(12)
      A = PRJ(10)*PHI*COSD(THETA)/R

      X =           R*SIND(A)
      Y = PRJ(13) - R*COSD(A)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE BONREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, DY, COSTHE, PHI, PRJ(0:20), R, THETA, X, Y

      DOUBLE PRECISION ATAN2D, COSD
*-----------------------------------------------------------------------
      IF (PRJ(1).EQ.0D0) THEN
*        Sanson-Flamsteed.
         CALL SFLREV (X, Y, PRJ, PHI, THETA, IERR)
         RETURN
      END IF

      IF (PRJ(11).NE.601D0) THEN
         CALL BONSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      DY = PRJ(13) - Y
      R = SQRT(X*X + DY*DY)
      IF (PRJ(1).LT.0D0) R = -R

      IF (R.EQ.0D0) THEN
         A = 0D0
      ELSE
         A = ATAN2D(X/R, DY/R)
      END IF

      THETA = (PRJ(13) - R)/PRJ(12)
      COSTHE = COSD(THETA)
      IF (COSTHE.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = A*(R/PRJ(10))/COSD(THETA)
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   PCO: polyconic projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/180)
*      PRJ(13)   1/r0
*      PRJ(14)   2*r0
*=======================================================================
      SUBROUTINE PCOSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 1D0
         PRJ(13) = 1D0
         PRJ(14) = 360D0/PI
      ELSE
         PRJ(12) = PRJ(10)*D2R
         PRJ(13) = 1D0/PRJ(12)
         PRJ(14) = 2D0*PRJ(10)
      END IF

      PRJ(11) = 602D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE PCOFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION A, COSTHE, COTTHE, PHI, PRJ(0:20), SINTHE,
     *          THETA, X, Y

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.602D0) THEN
         CALL PCOSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      COSTHE = COSD(THETA)
      SINTHE = SIND(THETA)
      A = PHI*SINTHE

      IF (SINTHE.EQ.0D0) THEN
         X = PRJ(12)*PHI
         Y = 0D0
      ELSE
         COTTHE = COSTHE/SINTHE
         X = PRJ(10)*COTTHE*SIND(A)
         Y = PRJ(10)*(COTTHE*(1D0 - COSD(A)) + THETA*D2R)
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE PCOREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR, J
      DOUBLE PRECISION F, FNEG, FPOS, LAMBDA, PHI, PRJ(0:20), TANTHE,
     *          THENEG, THEPOS, THETA, TOL, W, X, XP, XX, Y, YMTHE, YP

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ATAN2D, SIND, TAND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.602D0) THEN
         CALL PCOSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      W = ABS(Y*PRJ(13))
      IF (W.LT.TOL) THEN
         PHI = X*PRJ(13)
         THETA = 0D0
      ELSE IF (ABS(W-90D0).LT.TOL) THEN
         PHI = 0D0
         THETA = SIGN(90D0,Y)
      ELSE
*        Iterative solution using weighted division of the interval.
         IF (Y.GT.0D0) THEN
            THEPOS =  90D0
         ELSE
            THEPOS = -90D0
         END IF
         THENEG = 0D0

         XX = X*X
         YMTHE = Y - PRJ(12)*THEPOS
         FPOS = XX + YMTHE*YMTHE
         FNEG = -999D0

         DO 10 J = 1, 64
            IF (FNEG.LT.-100D0) THEN
*              Equal division of the interval.
               THETA = (THEPOS+THENEG)/2D0
            ELSE
*              Weighted division of the interval.
               LAMBDA = FPOS/(FPOS-FNEG)
               IF (LAMBDA.LT.0.1D0) THEN
                  LAMBDA = 0.1D0
               ELSE IF (LAMBDA.GT.0.9D0) THEN
                  LAMBDA = 0.9D0
               END IF
               THETA = THEPOS - LAMBDA*(THEPOS-THENEG)
            END IF

*           Compute the residue.
            YMTHE = Y - PRJ(12)*THETA
            TANTHE = TAND(THETA)
            F = XX + YMTHE*(YMTHE - PRJ(14)/TANTHE)

*           Check for convergence.
            IF (ABS(F).LT.TOL) GO TO 20
            IF (ABS(THEPOS-THENEG).LT.TOL) GO TO 20

*           Redefine the interval.
            IF (F.GT.0D0) THEN
               THEPOS = THETA
               FPOS = F
            ELSE
               THENEG = THETA
               FNEG = F
            END IF
 10      CONTINUE

 20      XP = PRJ(10) - YMTHE*TANTHE
         YP = X*TANTHE
         IF (XP.EQ.0D0 .AND. YP.EQ.0D0) THEN
            PHI = 0D0
         ELSE
            PHI = ATAN2D(YP, XP)/SIND(THETA)
         END IF
      END IF

      IERR = 0
      RETURN
      END

*=======================================================================
*   TSC: tangential spherical cube projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/4)
*      PRJ(13)   (4/pi)/r0
*=======================================================================
      SUBROUTINE TSCSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 45D0
         PRJ(13) = 1D0/45D0
      ELSE
         PRJ(12) = PRJ(10)*PI/4D0
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 701D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE TSCFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   FACE, IERR
      DOUBLE PRECISION COSTHE, L, M, N, PHI, PRJ(0:20), RHO, THETA, TOL,
     *          X, X0, XF, Y, Y0, YF

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.701D0) THEN
         CALL TSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      COSTHE = COSD(THETA)
      L = COSTHE*COSD(PHI)
      M = COSTHE*SIND(PHI)
      N = SIND(THETA)

      FACE = 0
      RHO  = N
      IF (L.GT.RHO) THEN
         FACE = 1
         RHO  = L
      END IF
      IF (M.GT.RHO) THEN
         FACE = 2
         RHO  = M
      END IF
      IF (-L.GT.RHO) THEN
         FACE = 3
         RHO  = -L
      END IF
      IF (-M.GT.RHO) THEN
         FACE = 4
         RHO  = -M
      END IF
      IF (-N.GT.RHO) THEN
         FACE = 5
         RHO  = -N
      END IF

      IF (FACE.EQ.0) THEN
         XF =  M/RHO
         YF = -L/RHO
         X0 =  0D0
         Y0 =  2D0
      ELSE IF (FACE.EQ.1) THEN
         XF =  M/RHO
         YF =  N/RHO
         X0 =  0D0
         Y0 =  0D0
      ELSE IF (FACE.EQ.2) THEN
         XF = -L/RHO
         YF =  N/RHO
         X0 =  2D0
         Y0 =  0D0
      ELSE IF (FACE.EQ.3) THEN
         XF = -M/RHO
         YF =  N/RHO
         X0 =  4D0
         Y0 =  0D0
      ELSE IF (FACE.EQ.4) THEN
         XF =  L/RHO
         YF =  N/RHO
         X0 =  6D0
         Y0 =  0D0
      ELSE IF (FACE.EQ.5) THEN
         XF =  M/RHO
         YF =  L/RHO
         X0 =  0D0
         Y0 = -2D0
      END IF

      IF (ABS(XF).GT.1D0) THEN
         IF (ABS(XF).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         XF = SIGN(1D0,XF)
      END IF
      IF (ABS(YF).GT.1D0) THEN
         IF (ABS(YF).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         YF = SIGN(1D0,YF)
      END IF

      X = PRJ(12)*(XF + X0)
      Y = PRJ(12)*(YF + Y0)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE TSCREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION L, M, N, PHI, PRJ(0:20), THETA, X, XF, Y, YF

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.701D0) THEN
         CALL TSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      XF = X*PRJ(13)
      YF = Y*PRJ(13)

*     Check bounds.
      IF (ABS(XF).LE.1D0) THEN
         IF (ABS(YF).GT.3D0) THEN
            IERR = 2
            RETURN
         END IF
      ELSE
         IF (ABS(XF).GT.7D0 .OR. ABS(YF).GT.1D0) THEN
            IERR = 2
            RETURN
         END IF
      END IF

*     Map negative faces to the other side.
      IF (XF.LT.-1D0) XF = XF + 8D0

*     Determine the face.
      IF (XF.GT.5D0) THEN
*        FACE = 4
         XF = XF - 6D0
         M  = -1D0/SQRT(1D0 + XF*XF + YF*YF)
         L  = -M*XF
         N  = -M*YF
      ELSE IF (XF.GT.3D0) THEN
*        FACE = 3
         XF = XF - 4D0
         L  = -1D0/SQRT(1D0 + XF*XF + YF*YF)
         M  =  L*XF
         N  = -L*YF
      ELSE IF (XF.GT.1D0) THEN
*        FACE = 2
         XF = XF - 2D0
         M  =  1D0/SQRT(1D0 + XF*XF + YF*YF)
         L  = -M*XF
         N  =  M*YF
      ELSE IF (YF.GT.1D0) THEN
*        FACE = 0
         YF = YF - 2D0
         N  = 1D0/SQRT(1D0 + XF*XF + YF*YF)
         L  = -N*YF
         M  =  N*XF
      ELSE IF (YF.LT.-1D0) THEN
*        FACE = 5
         YF = YF + 2D0
         N  = -1D0/SQRT(1D0 + XF*XF + YF*YF)
         L  = -N*YF
         M  = -N*XF
      ELSE
*        FACE = 1
         L  =  1D0/SQRT(1D0 + XF*XF + YF*YF)
         M  =  L*XF
         N  =  L*YF
      END IF

      IF (L.EQ.0D0 .AND. M.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(M, L)
      END IF
      THETA = ASIND(N)

      IERR = 0
      RETURN
      END

*=======================================================================
*   CSC: COBE quadrilateralized spherical cube projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/4)
*      PRJ(13)   (4/pi)/r0
*=======================================================================
      SUBROUTINE CSCSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 45D0
         PRJ(13) = 1D0/45D0
      ELSE
         PRJ(12) = PRJ(10)*PI/4D0
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 702D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CSCFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   FACE, IERR
      DOUBLE PRECISION COSTHE, ETA, L, M, N, PHI, PRJ(0:20), RHO, THETA,
     *          X, XI, Y

      REAL      A, A2, A2B2, A4, AB, B, B2, B4, CA2, CB2, TOL, X0, XF,
     *          Y0, YF
      REAL      C00, C10, C01, C11, C20, C02, D0, D1, MM, GAMMA, GSTAR,
     *          OMEGA1
      PARAMETER (GSTAR  =  1.37484847732)
      PARAMETER (MM     =  0.004869491981)
      PARAMETER (GAMMA  = -0.13161671474)
      PARAMETER (OMEGA1 = -0.159596235474)
      PARAMETER (D0  =  0.0759196200467)
      PARAMETER (D1  = -0.0217762490699)
      PARAMETER (C00 =  0.141189631152)
      PARAMETER (C10 =  0.0809701286525)
      PARAMETER (C01 = -0.281528535557)
      PARAMETER (C11 =  0.15384112876)
      PARAMETER (C20 = -0.178251207466)
      PARAMETER (C02 =  0.106959469314)

      PARAMETER (TOL = 1E-7)

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.702D0) THEN
         CALL CSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      COSTHE = COSD(THETA)
      L = COSTHE*COSD(PHI)
      M = COSTHE*SIND(PHI)
      N = SIND(THETA)

      FACE = 0
      RHO  = N
      IF (L.GT.RHO) THEN
         FACE = 1
         RHO  = L
      END IF
      IF (M.GT.RHO) THEN
         FACE = 2
         RHO  = M
      END IF
      IF (-L.GT.RHO) THEN
         FACE = 3
         RHO  = -L
      END IF
      IF (-M.GT.RHO) THEN
         FACE = 4
         RHO  = -M
      END IF
      IF (-N.GT.RHO) THEN
         FACE = 5
         RHO  = -N
      END IF

      IF (FACE.EQ.0) THEN
         XI  =  M
         ETA = -L
         X0  =  0.0
         Y0  =  2.0
      ELSE IF (FACE.EQ.1) THEN
         XI  =  M
         ETA =  N
         X0  =  0.0
         Y0  =  0.0
      ELSE IF (FACE.EQ.2) THEN
         XI  = -L
         ETA =  N
         X0  =  2.0
         Y0  =  0.0
      ELSE IF (FACE.EQ.3) THEN
         XI  = -M
         ETA =  N
         X0  =  4.0
         Y0  =  0.0
      ELSE IF (FACE.EQ.4) THEN
         XI  =  L
         ETA =  N
         X0  =  6.0
         Y0  =  0.0
      ELSE IF (FACE.EQ.5) THEN
         XI  =  M
         ETA =  L
         X0  =  0.0
         Y0  = -2.0
      END IF

      A =  XI/RHO
      B = ETA/RHO

      A2 = A*A
      B2 = B*B
      CA2 = 1.0 - A2
      CB2 = 1.0 - B2

*     Avoid floating underflows.
      A4 = 0.0
      B4 = 0.0
      AB = ABS(A*B)
      A2B2 = 0.0
      IF (A2.GT.1E-16) A4 = A2*A2
      IF (B2.GT.1E-16) B4 = B2*B2
      IF (AB.GT.1E-16) A2B2 = AB*AB

      XF = A*(A2 + CA2*(GSTAR + B2*(GAMMA*CA2 + MM*A2 +
     *       CB2*(C00 + C10*A2 + C01*B2 + C11*A2B2 + C20*A4 + C02*B4)) +
     *       A2*(OMEGA1 - CA2*(D0 + D1*A2))))
      YF = B*(B2 + CB2*(GSTAR + A2*(GAMMA*CB2 + MM*B2 +
     *       CA2*(C00 + C10*B2 + C01*A2 + C11*A2B2 + C20*B4 + C02*A4)) +
     *       B2*(OMEGA1 - CB2*(D0 + D1*B2))))

      IF (ABS(XF).GT.1.0) THEN
         IF (ABS(XF).GT.1.0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         XF = SIGN(1.0,XF)
      END IF
      IF (ABS(YF).GT.1.0) THEN
         IF (ABS(YF).GT.1.0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         YF = SIGN(1.0,YF)
      END IF

      X = PRJ(12)*(X0 + XF)
      Y = PRJ(12)*(Y0 + YF)

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE CSCREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   FACE, IERR
      DOUBLE PRECISION L, M, N, PHI, PRJ(0:20), THETA, X, Y

      REAL      A, B, XF, XX, YF, YY
      REAL      P00, P01, P02, P03, P04, P05, P06, P10, P11, P12,
     *          P13, P14, P15, P20, P21, P22, P23, P24, P30, P31, P32,
     *          P33, P40, P41, P42, P50, P51, P60
      PARAMETER (P00 = -0.27292696)
      PARAMETER (P10 = -0.07629969)
      PARAMETER (P20 = -0.22797056)
      PARAMETER (P30 =  0.54852384)
      PARAMETER (P40 = -0.62930065)
      PARAMETER (P50 =  0.25795794)
      PARAMETER (P60 =  0.02584375)
      PARAMETER (P01 = -0.02819452)
      PARAMETER (P11 = -0.01471565)
      PARAMETER (P21 =  0.48051509)
      PARAMETER (P31 = -1.74114454)
      PARAMETER (P41 =  1.71547508)
      PARAMETER (P51 = -0.53022337)
      PARAMETER (P02 =  0.27058160)
      PARAMETER (P12 = -0.56800938)
      PARAMETER (P22 =  0.30803317)
      PARAMETER (P32 =  0.98938102)
      PARAMETER (P42 = -0.83180469)
      PARAMETER (P03 = -0.60441560)
      PARAMETER (P13 =  1.50880086)
      PARAMETER (P23 = -0.93678576)
      PARAMETER (P33 =  0.08693841)
      PARAMETER (P04 =  0.93412077)
      PARAMETER (P14 = -1.41601920)
      PARAMETER (P24 =  0.33887446)
      PARAMETER (P05 = -0.63915306)
      PARAMETER (P15 =  0.52032238)
      PARAMETER (P06 =  0.14381585)

      DOUBLE PRECISION ASIND, ATAN2D
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.702D0) THEN
         CALL CSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      XF = X*PRJ(13)
      YF = Y*PRJ(13)

*     Check bounds.
      IF (ABS(XF).LE.1D0) THEN
         IF (ABS(YF).GT.3D0) THEN
            IERR = 2
            RETURN
         END IF
      ELSE
         IF (ABS(XF).GT.7D0 .OR. ABS(YF).GT.1D0) THEN
            IERR = 2
            RETURN
         END IF
      END IF

*     Map negative faces to the other side.
      IF (XF.LT.-1D0) XF = XF + 8D0

*     Determine the face.
      IF (XF.GT.5.0) THEN
         FACE = 4
         XF = XF - 6.0
      ELSE IF (XF.GT.3.0) THEN
         FACE = 3
         XF = XF - 4.0
      ELSE IF (XF.GT.1.0) THEN
         FACE = 2
         XF = XF - 2.0
      ELSE IF (YF.GT.1.0) THEN
         FACE = 0
         YF = YF - 2.0
      ELSE IF (YF.LT.-1.0) THEN
         FACE = 5
         YF = YF + 2.0
      ELSE
         FACE = 1
      END IF

      XX  =  XF*XF
      YY  =  YF*YF

      A =   (P00+XX*(P10+XX*(P20+XX*(P30+XX*(P40+XX*(P50+XX*(P60)))))) +
     *   YY*(P01+XX*(P11+XX*(P21+XX*(P31+XX*(P41+XX*(P51))))) +
     *   YY*(P02+XX*(P12+XX*(P22+XX*(P32+XX*(P42)))) +
     *   YY*(P03+XX*(P13+XX*(P23+XX*(P33))) +
     *   YY*(P04+XX*(P14+XX*(P24)) +
     *   YY*(P05+XX*(P15) +
     *   YY*(P06)))))))
      A = XF + XF*(1.0 - XX)*A

      B =   (P00+YY*(P10+YY*(P20+YY*(P30+YY*(P40+YY*(P50+YY*(P60)))))) +
     *   XX*(P01+YY*(P11+YY*(P21+YY*(P31+YY*(P41+YY*(P51))))) +
     *   XX*(P02+YY*(P12+YY*(P22+YY*(P32+YY*(P42)))) +
     *   XX*(P03+YY*(P13+YY*(P23+YY*(P33))) +
     *   XX*(P04+YY*(P14+YY*(P24)) +
     *   XX*(P05+YY*(P15) +
     *   XX*(P06)))))))
      B = YF + YF*(1.0 - YY)*B

      IF (FACE.EQ.0) THEN
         N =  1D0/SQRT(A*A + B*B + 1D0)
         L = -B*N
         M =  A*N
      ELSE IF (FACE.EQ.1) THEN
         L =  1D0/SQRT(A*A + B*B + 1D0)
         M =  A*L
         N =  B*L
      ELSE IF (FACE.EQ.2) THEN
         M =  1D0/SQRT(A*A + B*B + 1D0)
         L = -A*M
         N =  B*M
      ELSE IF (FACE.EQ.3) THEN
         L = -1D0/SQRT(A*A + B*B + 1D0)
         M =  A*L
         N = -B*L
      ELSE IF (FACE.EQ.4) THEN
         M = -1D0/SQRT(A*A + B*B + 1D0)
         L = -A*M
         N = -B*M
      ELSE IF (FACE.EQ.5) THEN
         N = -1D0/SQRT(A*A + B*B + 1D0)
         L = -B*N
         M = -A*N
      END IF

      IF (L.EQ.0D0 .AND. M.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(M, L)
      END IF
      THETA = ASIND(N)

      IERR = 0
      RETURN
      END

*=======================================================================
*   QSC: quadrilaterilized spherical cube projection.
*
*   Given and/or returned:
*      PRJ(10)   r0; reset to 180/pi if 0.
*      PRJ(11)   State flag.
*      PRJ(12)   r0*(pi/4)
*      PRJ(13)   (4/pi)/r0
*=======================================================================
      SUBROUTINE QSCSET (PRJ, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION PRJ(0:20)

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)
*-----------------------------------------------------------------------
      IF (PRJ(10).EQ.0D0) THEN
         PRJ(10) = R2D
         PRJ(12) = 45D0
         PRJ(13) = 1D0/45D0
      ELSE
         PRJ(12) = PRJ(10)*PI/4D0
         PRJ(13) = 1D0/PRJ(12)
      END IF

      PRJ(11) = 703D0

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE QSCFWD (PHI, THETA, PRJ, X, Y, IERR)
*-----------------------------------------------------------------------
      INTEGER   FACE, IERR
      DOUBLE PRECISION CHI, COSTHE, ETA, L, M, N, P, PHI, PRJ(0:20),
     *          PSI, RHO, RHU, T, THETA, TOL, X, X0, XF, XI, Y, Y0, YF

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION ASIND, ATAND, COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.703D0) THEN
         CALL QSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      IF (ABS(THETA).EQ.90D0) THEN
         X = 0D0
         Y = SIGN(2D0*PRJ(12),THETA)
         IERR = 0
         RETURN
      END IF

      COSTHE = COSD(THETA)
      L = COSTHE*COSD(PHI)
      M = COSTHE*SIND(PHI)
      N = SIND(THETA)

      FACE = 0
      RHO  = N
      IF (L.GT.RHO) THEN
         FACE = 1
         RHO  = L
      END IF
      IF (M.GT.RHO) THEN
         FACE = 2
         RHO  = M
      END IF
      IF (-L.GT.RHO) THEN
         FACE = 3
         RHO  = -L
      END IF
      IF (-M.GT.RHO) THEN
         FACE = 4
         RHO  = -M
      END IF
      IF (-N.GT.RHO) THEN
         FACE = 5
         RHO  = -N
      END IF

      RHU = 1D0 - RHO

      IF (FACE.EQ.0) THEN
         XI  =  M
         ETA = -L
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = (90D0 - THETA)*D2R
            RHU = T*T/2D0
         END IF
         X0  =  0D0
         Y0  =  2D0
      ELSE IF (FACE.EQ.1) THEN
         XI  =  M
         ETA =  N
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = THETA*D2R
            P = MOD(PHI,360D0)
            IF (P.LT.-180D0) P = P + 360D0
            IF (P.GT.+180D0) P = P - 360D0
            P = P*D2R
            RHU = (P*P + T*T)/2D0
         END IF
         X0  =  0D0
         Y0  =  0D0
      ELSE IF (FACE.EQ.2) THEN
         XI  = -L
         ETA =  N
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = THETA*D2R
            P = MOD(PHI,360D0)
            IF (P.LT.-180D0) P = P + 360D0
            P = (90D0 - P)*D2R
            RHU = (P*P + T*T)/2D0
         END IF
         X0  =  2D0
         Y0  =  0D0
      ELSE IF (FACE.EQ.3) THEN
         XI  = -M
         ETA =  N
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = THETA*D2R
            P = MOD(PHI,360D0)
            IF (P.LT.0D0) P = P + 360D0
            P = (180D0 - P)*D2R
            RHU = (P*P + T*T)/2D0
         END IF
         X0  =  4D0
         Y0  =  0D0
      ELSE IF (FACE.EQ.4) THEN
         XI  =  L
         ETA =  N
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = THETA*D2R
            P = MOD(PHI,360D0)
            IF (P.GT.+180D0) P = P - 360D0
            P = (90D0 + P)*D2R
            RHU = (P*P + T*T)/2D0
         END IF
         X0  =  6
         Y0  =  0D0
      ELSE IF (FACE.EQ.5) THEN
         XI  =  M
         ETA =  L
         IF (RHU.LT.1D-8) THEN
*           Small angle formula.
            T = (90D0 + THETA)*D2R
            RHU = T*T/2D0
         END IF
         X0  =  0D0
         Y0  = -2
      END IF

      IF (XI.EQ.0D0 .AND. ETA.EQ.0D0) THEN
         XF  = 0D0
         YF  = 0D0
      ELSE IF (-XI.GE.ABS(ETA)) THEN
         PSI = ETA/XI
         CHI = 1D0 + PSI*PSI
         XF  = -SQRT(RHU/(1D0-1D0/SQRT(1D0+CHI)))
         YF  = (XF/15D0)*(ATAND(PSI) - ASIND(PSI/SQRT(CHI+CHI)))
      ELSE IF (XI.GE.ABS(ETA)) THEN
         PSI = ETA/XI
         CHI = 1D0 + PSI*PSI
         XF  =  SQRT(RHU/(1D0-1D0/SQRT(1D0+CHI)))
         YF  = (XF/15D0)*(ATAND(PSI) - ASIND(PSI/SQRT(CHI+CHI)))
      ELSE IF (-ETA.GT.ABS(XI)) THEN
         PSI = XI/ETA
         CHI = 1D0 + PSI*PSI
         YF  = -SQRT(RHU/(1D0-1D0/SQRT(1D0+CHI)))
         XF  = (YF/15D0)*(ATAND(PSI) - ASIND(PSI/SQRT(CHI+CHI)))
      ELSE IF (ETA.GT.ABS(XI)) THEN
         PSI = XI/ETA
         CHI = 1D0 + PSI*PSI
         YF  =  SQRT(RHU/(1D0-1D0/SQRT(1D0+CHI)))
         XF  = (YF/15D0)*(ATAND(PSI) - ASIND(PSI/SQRT(CHI+CHI)))
      END IF

      IF (ABS(XF).GT.1D0) THEN
         IF (ABS(XF).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         XF = SIGN(1D0,XF)
      END IF
      IF (ABS(YF).GT.1D0) THEN
         IF (ABS(YF).GT.1D0+TOL) THEN
            IERR = 2
            RETURN
         END IF
         YF = SIGN(1D0,YF)
      END IF

      X = PRJ(12)*(XF + X0)
      Y = PRJ(12)*(YF + Y0)


      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE QSCREV (X, Y, PRJ, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      LOGICAL   DIRECT
      INTEGER   FACE, IERR
      DOUBLE PRECISION CHI, L, M, N, PHI, PRJ(0:20), PSI, RHO, RHU,
     *          THETA, TOL, X, XF, Y, YF, W

      PARAMETER (TOL = 1D-12)

      DOUBLE PRECISION C2
      PARAMETER (C2 = 1D0/1.4142135623730950488D0)

      DOUBLE PRECISION ASIND, ATAN2D, COSD, SIND
*-----------------------------------------------------------------------
      IF (PRJ(11).NE.703D0) THEN
         CALL QSCSET (PRJ, IERR)
         IF (IERR.NE.0) RETURN
      END IF

      XF = X*PRJ(13)
      YF = Y*PRJ(13)

*     Check bounds.
      IF (ABS(XF).LE.1D0) THEN
         IF (ABS(YF).GT.3D0) THEN
            IERR = 2
            RETURN
         END IF
      ELSE
         IF (ABS(XF).GT.7D0 .OR. ABS(YF).GT.1D0) THEN
            IERR = 2
            RETURN
         END IF
      END IF

*     Map negative faces to the other side.
      IF (XF.LT.-1D0) XF = XF + 8D0

*     Determine the face.
      IF (XF.GT.5D0) THEN
         FACE = 4
         XF = XF - 6D0
      ELSE IF (XF.GT.3D0) THEN
         FACE = 3
         XF = XF - 4D0
      ELSE IF (XF.GT.1D0) THEN
         FACE = 2
         XF = XF - 2D0
      ELSE IF (YF.GT.1D0) THEN
         FACE = 0
         YF = YF - 2D0
      ELSE IF (YF.LT.-1D0) THEN
         FACE = 5
         YF = YF + 2D0
      ELSE
         FACE = 1
      END IF

      DIRECT = ABS(XF).GT.ABS(YF)
      IF (DIRECT) THEN
         IF (XF.EQ.0D0) THEN
            PSI = 0D0
            CHI = 1D0
            RHO = 1D0
            RHU = 0D0
         ELSE
            W = 15D0*YF/XF
            PSI = SIND(W)/(COSD(W) - C2)
            CHI = 1D0 + PSI*PSI
            RHU = XF*XF*(1D0 - 1D0/SQRT(1D0 + CHI))
            RHO = 1D0 - RHU
         END IF
      ELSE
         IF (YF.EQ.0D0) THEN
            PSI = 0D0
            CHI = 1D0
            RHO = 1D0
            RHU = 0D0
         ELSE
            W = 15D0*XF/YF
            PSI = SIND(W)/(COSD(W) - C2)
            CHI = 1D0 + PSI*PSI
            RHU = YF*YF*(1D0 - 1D0/SQRT(1D0 + CHI))
            RHO = 1D0 - RHU
         END IF
      END IF

      IF (RHO.LT.-1D0) THEN
         IF (RHO.LT.-1D0-TOL) THEN
            IERR = 2
            RETURN
         END IF

         RHO = -1D0
         RHU =  2D0
         W   =  0D0
      ELSE
         W = SQRT(RHU*(2D0-RHU)/CHI)
      END IF

      IF (FACE.EQ.0) THEN
         N = RHO
         IF (DIRECT) THEN
            M = W
            IF (XF.LT.0D0) M = -M
            L = -M*PSI
         ELSE
            L = W
            IF (YF.GT.0D0) L = -L
            M = -L*PSI
         END IF
      ELSE IF (FACE.EQ.1) THEN
         L = RHO
         IF (DIRECT) THEN
            M = W
            IF (XF.LT.0D0) M = -M
            N = M*PSI
         ELSE
            N = W
            IF (YF.LT.0D0) N = -N
            M = N*PSI
         END IF
      ELSE IF (FACE.EQ.2) THEN
         M =  RHO
         IF (DIRECT) THEN
            L = W
            IF (XF.GT.0D0) L = -L
            N = -L*PSI
         ELSE
            N = W
            IF (YF.LT.0D0) N = -N
            L = -N*PSI
         END IF
      ELSE IF (FACE.EQ.3) THEN
         L = -RHO
         IF (DIRECT) THEN
            M = W
            IF (XF.GT.0D0) M = -M
            N = -M*PSI
         ELSE
            N = W
            IF (YF.LT.0D0) N = -N
            M = -N*PSI
         END IF
      ELSE IF (FACE.EQ.4) THEN
         M = -RHO
         IF (DIRECT) THEN
            L = W
            IF (XF.LT.0D0) L = -L
            N = L*PSI
         ELSE
            N = W
            IF (YF.LT.0D0) N = -N
            L = N*PSI
         END IF
      ELSE IF (FACE.EQ.5) THEN
         N = -RHO
         IF (DIRECT) THEN
            M = W
            IF (XF.LT.0D0) M = -M
            L = M*PSI
         ELSE
            L = W
            IF (YF.LT.0D0) L = -L
            M = L*PSI
         END IF
      END IF

      IF (L.EQ.0D0 .AND. M.EQ.0D0) THEN
         PHI = 0D0
      ELSE
         PHI = ATAN2D(M, L)
      END IF
      THETA = ASIND(N)

      IERR = 0
      RETURN
      END
