*=======================================================================
*
*   PGSBOX - a non-linear coordinate axis plotter for PGPLOT.
*   Copyright (C) 1997-2000, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning PGSBOX may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   PGWCSL defines curvilinear celestial coordinate systems for PGSBOX
*   using WCSLIB.
*
*   Given:
*      OPCODE   I        Transformation code:
*                           +2: Compute a set of pixel coordinates which
*                               describe a path between this and the
*                               previous pair of world coordinates
*                               remembered from the last call with
*                               OPCODE = +1 or +2.
*                           +1: Compute pixel coordinates from world
*                               coordinates.
*                            0: Initialize.
*                           -1: Compute world coordinates from pixel
*                               coordinates.
*
*      NLC      I        Number of elements in CTYPE (=16).
*
*      NLI      I        Number of elements in WCS (=4).
*
*      NLD      I        Number of elements in NLDPRM (=50).
*
*      CTYPE    C(NLC)*1 Character array (contains the CTYPE array,
*                        see below).
*
*   Given and/or returned:
*      WCS      I(NLI)   Integer coefficients (contains the WCS
*                        array, see below).
*
*      NLDPRM   D(NLD)   Double precision coefficients (contains the
*                        LIN, PRJ and CEL arrays, see below).
*
*      WORLD    D(2)     World coordinates.  WORLD(1) and WORLD(2)
*                        are the longitude and latitude, in degrees.
*                        Given if OPCODE > 0, returned if OPCODE < 0.
*
*      PIXEL    D(2)     Pixel coordinates.
*                        Given if OPCODE < 0, returned if OPCODE > 0.
*
*      CONTRL   I        Control flag for OPCODE = +2:
*                           0: Normal state
*                           1: A discontinuity has been encountered;
*                              force PGSBOX to flush its plotting buffer
*                              and call PGWCSL again with the same world
*                              coordinates.
*                           2: Call PGWCSL again with the same world
*                              coordinates.
*
*      CONTXT   D(20)    Context elements for OPCODE = +2.
*
*   Returned:
*      IERR     I        Error status
*                           0: Success.
*                           1: Invalid parameters.
*                           2: Invalid world coordinate.
*                           3: Invalid pixel coordinate.
*
*   Notes
*   -----
*    1) PGWCSL assumes a simple 2-D image.  The CTYPE array must be
*       constructed by the caller as follows:
*
*                FITS
*       CTYPE   keyword
*
*        (1:8)   CTYPE1
*        (9:16)  CTYPE2
*
*       The WCS array is maintained by PGWCSL and WCSLIB and must not
*       be disturbed by the caller after initialization with OPCODE=0.
*
*                WCS      FITS
*         WCS   array    keyword
*
*         (1)   WCS(0)     -
*         (2)   WCS(1)     -
*         (3)   WCS(2)     -
*         (4)   WCS(3)     -
*
*       The NLDPRM array is constructed as follows:
*
*                WCS      FITS
*       NLDPRM  array    keyword
*
*         (1)   LIN(1)     -
*         (2)   LIN(2)     -
*         (3)   LIN(3)     -
*         (4)   LIN(4)   CRPIX1
*         (5)   LIN(5)   CRPIX2
*         (6)   LIN(6)   PC001001
*         (7)   LIN(7)   PC002001
*         (8)   LIN(8)   PC001002
*         (9)   LIN(9)   PC002002
*        (10)   LIN(10)  CDELT1
*        (11)   LIN(11)  CDELT2
*        (12)   LIN(12)    -
*        (13)   LIN(13)    -
*        (14)   LIN(14)    -
*        (15)   LIN(15)    -
*        (16)   LIN(16)    -
*        (17)   LIN(17)    -
*        (18)   LIN(18)    -
*        (19)   LIN(19)    -
*        (20)   CEL(1)   CRVAL1
*        (21)   CEL(2)   CRVAL2
*        (22)   CEL(3)   LONGPOLE
*        (23)   CEL(4)   LATPOLE
*        (24)   CEL(5)     -
*        (25)   CEL(6)     -
*        (26)   CEL(7)     -
*        (27)   CEL(8)     -
*        (28)   CEL(9)     -
*        (29)   CEL(10)    -
*        (30)   PRJ(0)   PROJP0
*        (31)   PRJ(1)   PROJP1
*        (32)   PRJ(2)   PROJP2
*        (33)   PRJ(3)   PROJP3
*        (34)   PRJ(4)   PROJP4
*        (35)   PRJ(5)   PROJP5
*        (36)   PRJ(6)   PROJP6
*        (37)   PRJ(7)   PROJP7
*        (38)   PRJ(8)   PROJP8
*        (39)   PRJ(9)   PROJP9
*        (40)   PRJ(10)    -
*        (41)   PRJ(11)    -
*        (42)   PRJ(12)    -
*        (43)   PRJ(13)    -
*        (44)   PRJ(14)    -
*        (45)   PRJ(15)    -
*        (46)   PRJ(16)    -
*        (47)   PRJ(17)    -
*        (48)   PRJ(18)    -
*        (49)   PRJ(19)    -
*        (50)   PRJ(20)    -
*
*       Elements of the array which correspond to FITS header cards must
*       be given by the caller.  The other elements are maintained by
*       PGWCSL and WCSLIB as described in the notes provided with
*       WCSLIB.  NLDPRM must not be disturbed by the caller after
*       initialization with OPCODE=0.
*
*    2) PGWCSL doesn't properly handle discontinuities between the faces
*       of the quadcube projections.
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: pgwcsl.f,v 1.12 2001/02/19 02:18:18 mcalabre Exp $
*=======================================================================
      SUBROUTINE PGWCSL (OPCODE, NLC, NLI, NLD, CTYPE, WCS, NLDPRM,
     :   WORLD, PIXEL, CONTRL, CONTXT, IERR)
*-----------------------------------------------------------------------
      INTEGER   CONTRL, IERR, NLC, NLD, NLI, OPCODE, WCS(NLI)
      DOUBLE PRECISION CONTXT(20), DP, DUMMY(2), IMGCRD(2), LAT, LNG,
     :          NLDPRM(NLD), PH, PHI, PIXEL(2), SDUMMY, TH, THETA,
     :          WORLD(2), WRLD(2)
      CHARACTER CTYPE(NLC)*1
*-----------------------------------------------------------------------
      IERR = 0

      IF (OPCODE.EQ.2) THEN
*        Compute pixel coordinates from world coordinates.
         WRLD(WCS(2)) = WORLD(1)
         IF (WORLD(2).GT.90D0) THEN
            WRLD(WCS(3)) = 90D0
         ELSE IF (WORLD(2).LT.-90D0) THEN
            WRLD(WCS(3)) = -90D0
         ELSE
            WRLD(WCS(3)) = WORLD(2)
         END IF

         IF (CONTRL.EQ.0) THEN
            CALL WCSFWD (CTYPE, WCS, WRLD, DUMMY, NLDPRM(20), PHI,
     :         THETA, NLDPRM(30), IMGCRD, NLDPRM(1), PIXEL, IERR)
            IF (IERR.NE.0) THEN
*              Translate error codes.
               IF (IERR.EQ.2) THEN
                  IERR = 1
               ELSE IF (IERR.EQ.3) THEN
                  IERR = 2
               ELSE IF (IERR.EQ.4) THEN
                  IERR = 1
               END IF
               RETURN
            END IF

            IF (ABS(PHI-CONTXT(3)).GT.180D0) THEN
*              Hit a discontinuity at PHI = +/- 180.
               CONTXT(5) = PIXEL(1)
               CONTXT(6) = PIXEL(2)

               IF (CONTXT(3).GT.PHI) THEN
                  PH = 179.9999D0
                  DP = (PHI - CONTXT(3)) + 360D0
               ELSE
                  PH = -179.9999D0
                  DP = (PHI - CONTXT(3)) - 360D0
               END IF

*              First approximation for theta.
               IF (DP.EQ.0D0) THEN
                  TH = CONTXT(4)
               ELSE
                  TH = CONTXT(4) + (PH-CONTXT(3))*(THETA-CONTXT(4))/DP
               END IF

*              Iterate once to refine the value of theta.
               CALL SPHREV (PH, TH, NLDPRM(25), LNG, LAT, IERR)
               IF (WRLD(1).EQ.CONTXT(1)) THEN
*                 We are following a meridian of longitude.
                  LNG = WRLD(1)
               ELSE
*                 We are following a parallel of latitude.
                  LAT = WRLD(2)
               END IF
               CALL SPHFWD (LNG, LAT, NLDPRM(25), SDUMMY, TH, IERR)

               CONTXT(1) = WRLD(1)
               CONTXT(2) = WRLD(2)
               CONTXT(3) = PHI
               CONTXT(4) = THETA

*              Pixel coordinates crossing into the discontinuity.
               CALL SPHREV (PH, TH, NLDPRM(25), WRLD(1), WRLD(2), IERR)
               CALL WCSFWD (CTYPE, WCS, WRLD, DUMMY, NLDPRM(20), PHI,
     :            THETA, NLDPRM(30), IMGCRD, NLDPRM(1), PIXEL, IERR)
               IF (IERR.NE.0) THEN
*                 Translate error codes.
                  IF (IERR.EQ.2) THEN
                     IERR = 1
                  ELSE IF (IERR.EQ.3) THEN
                     IERR = 2
                  ELSE IF (IERR.EQ.4) THEN
                     IERR = 1
                  END IF
                  RETURN
               END IF

*              Pixel coordinates crossing out of the discontinuity.
               CALL SPHREV (-PH, TH, NLDPRM(25), WRLD(1), WRLD(2), IERR)
               CALL WCSFWD (CTYPE, WCS, WRLD, DUMMY, NLDPRM(20), PHI,
     :            THETA, NLDPRM(30), IMGCRD, NLDPRM(1), CONTXT(7), IERR)
               IF (IERR.NE.0) THEN
*                 Translate error codes.
                  IF (IERR.EQ.2) THEN
                     IERR = 1
                  ELSE IF (IERR.EQ.3) THEN
                     IERR = 2
                  ELSE IF (IERR.EQ.4) THEN
                     IERR = 1
                  END IF
                  RETURN
               END IF

               CONTRL = 1
            ELSE
*              Normal mode, no discontinuity.
               CONTXT(1) = WRLD(1)
               CONTXT(2) = WRLD(2)
               CONTXT(3) = PHI
               CONTXT(4) = THETA
            END IF
         ELSE
            IF (CONTRL.EQ.1) THEN
*              Move to the other side of the discontinuity.
               PIXEL(1) = CONTXT(7)
               PIXEL(2) = CONTXT(8)
               CONTRL = 2
            ELSE
*              Complete the traversal.
               PIXEL(1) = CONTXT(5)
               PIXEL(2) = CONTXT(6)
               CONTRL = 0
            END IF
         END IF

      ELSE IF (OPCODE.EQ.1) THEN
*        Compute pixel coordinates from world coordinates.
         WRLD(WCS(2)) = WORLD(1)
         IF (WORLD(2).GT.90D0) THEN
            WRLD(WCS(3)) = 90D0
         ELSE IF (WORLD(2).LT.-90D0) THEN
            WRLD(WCS(3)) = -90D0
         ELSE
            WRLD(WCS(3)) = WORLD(2)
         END IF

         CALL WCSFWD (CTYPE, WCS, WRLD, DUMMY, NLDPRM(20),
     :      PHI, THETA, NLDPRM(30), IMGCRD, NLDPRM(1), PIXEL, IERR)
         IF (IERR.NE.0) THEN
*           Translate error codes.
            IF (IERR.EQ.2 .OR. IERR.EQ.4) THEN
               IERR = 1
            ELSE IF (IERR.EQ.3) THEN
               IERR = 2
            END IF
            RETURN
         END IF

         CONTXT(1) = WRLD(1)
         CONTXT(2) = WRLD(2)
         CONTXT(3) = PHI
         CONTXT(4) = THETA

      ELSE IF (OPCODE.EQ.0) THEN
*        Initialize.
         IF (NLC.LT.16 .OR. NLI.LT.4 .OR. NLD.LT.50) THEN
            IERR = 1
            RETURN
         END IF

         WCS(1)  = 0
         NLDPRM(1)  = 0D0
         NLDPRM(2)  = 2D0
         NLDPRM(3)  = 2D0
         NLDPRM(24) = 0D0
         NLDPRM(41) = 0D0
         CALL WCSSET (2, CTYPE, WCS, IERR)

         CONTRL = 0

      ELSE IF (OPCODE.EQ.-1) THEN
*        Compute world coordinates from pixel coordinates.
         CALL WCSREV (CTYPE, WCS, PIXEL, NLDPRM(1), IMGCRD,
     :      NLDPRM(30), PHI, THETA, DUMMY, NLDPRM(20), WRLD, IERR)
         IF (IERR.NE.0) THEN
*           Translate error codes.
            IF (IERR.EQ.2 .OR. IERR.EQ.4) IERR = 1
            RETURN
         END IF

         WORLD(1) = WRLD(WCS(2))
         WORLD(2) = WRLD(WCS(3))

         IF (PHI.LT.-180D0 .OR. PHI.GT.180D0) THEN
*           Pixel lies outside the principle range of native longitude.
            IERR = 3
            RETURN
         END IF

      ELSE
         IERR = 1
      END IF


      RETURN
      END
