*=======================================================================
*
*   PGSBOX - a non-linear coordinate axis plotter for PGPLOT.
*   Copyright (C) 1997-2001, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning PGSBOX may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: pgtest.f,v 1.21 2002/04/03 06:21:31 mcalabre Exp $
*=======================================================================
      PROGRAM PGTEST
*=======================================================================
      LOGICAL   LARGE
      INTEGER   AXEN(2), C0(7), CI(7), GCODE(2), IC, IERR, J, NLIPRM(2),
     :          WCS(4)
      REAL      BLC(2), SCL, TRC(2)
      DOUBLE PRECISION CDELT(2), CACHE(4,0:256), CRPIX(2), CRVAL(2),
     :          GRID1(0:10), GRID2(0:10), LATPOL, LIN(19), LNGPOL,
     :          NLDPRM(50), PC(2,2), PROJP(0:9), TIKLEN
      CHARACTER CTYPE(2)*8, DEVTYP*16, ESCAPE*1, FCODE(2)*4,
     :          IDENTS(3)*80, OPT(2)*1

*     From the prologue of PGWCSL.
      EQUIVALENCE (LIN(4),  CRPIX)
      EQUIVALENCE (LIN(6),  PC)
      EQUIVALENCE (LIN(10), CDELT)
      EQUIVALENCE (NLDPRM(1),  LIN)
      EQUIVALENCE (NLDPRM(20), CRVAL)
      EQUIVALENCE (NLDPRM(22), LNGPOL)
      EQUIVALENCE (NLDPRM(23), LATPOL)
      EQUIVALENCE (NLDPRM(30), PROJP)

      EXTERNAL LNGVEL
      EXTERNAL FSCAN
      EXTERNAL PGCRFN
      EXTERNAL PGWCSL

*     A portability fix - does '\' itself need to be escaped?
      DATA ESCAPE /'\\'/

      DOUBLE PRECISION COSD, SIND
*-----------------------------------------------------------------------
*     Uncomment the following two lines to raise SIGFPE on floating
*     point exceptions for the Sun FORTRAN compiler.  This signal can
*     be caught within 'dbx' by issuing the command "catch FPE".
*#include <f77_floatingpoint.h>
*      call ieee_handler ('set', 'common', SIGFPE_ABORT)
*-----------------------------------------------------------------------
*     Setup.
      AXEN(1) = 512
      AXEN(2) = 512

      BLC(1) = 0.5
      BLC(2) = 0.5
      TRC(1) = AXEN(1) + 0.5
      TRC(2) = AXEN(2) + 0.5

      CALL PGBEG (0, '?', 1, 1)

      CALL PGQINF ('TYPE', DEVTYP, J)
      IF (DEVTYP.EQ.'PS'  .OR.
     :    DEVTYP.EQ.'VPS' .OR.
     :    DEVTYP.EQ.'CPS' .OR.
     :    DEVTYP.EQ.'VCPS') THEN
*        Switch black and white.
         CALL PGSCR (0, 1.0, 1.0, 1.0)
         CALL PGSCR (1, 0.0, 0.0, 0.0)
      END IF

      LARGE = DEVTYP.EQ.'XWINDOWS'
      IF (LARGE) THEN
         SCL = 1.0
         CALL PGVSTD ()
      ELSE
         SCL = 0.7
         CALL PGVSIZ (1.0, 3.0, 1.0, 3.0)
      END IF

*     Yellow.
      CALL PGSCR (2, 1.0, 1.0, 0.0)
*     White.
      CALL PGSCR (3, 1.0, 1.0, 1.0)
*     Pale blue.
      CALL PGSCR (4, 0.5, 0.5, 0.8)
*     Pale red.
      CALL PGSCR (5, 0.8, 0.5, 0.5)
*     Grey.
      CALL PGSCR (6, 0.7, 0.7, 0.7)
*     Dark green.
      CALL PGSCR (7, 0.3, 0.5, 0.3)

      C0(1) = -1
      C0(2) = -1
      C0(3) = -1
      C0(4) = -1
      C0(5) = -1
      C0(6) = -1
      C0(7) = -1

      CALL PGWNAD (0.0, 1.0, 0.0, 1.0)
      CALL PGASK (.TRUE.)
      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     First example: longitude-velocity map.  The y-axis is regularly
*     spaced in frequency but is to be labelled as a true relativistic
*     velocity.  PGSBOX uses subroutine LNGVEL.  Features:
*        - Separable (i.e. orthogonal), non-linear coordinate system.
*        - Automatic choice of coordinate increments.
*        - Extraction of a common scaling factor.
*        - Automatic choice of what edges to label.
*        - Request for tickmarks (internal) for one coordinate and grid
*          lines for the other.
*        - Simple two-colour grid using two calls with deferred
*          labelling on the first call.
*        - Degree labelling.
*        - Suppression of zero arcmin and arcsec fields in sexagesimal
*          degree format.

*     Reference pixel coordinates.
      NLDPRM(1) = 1D0
      NLDPRM(2) = 256D0

*     Reference pixel values.
      NLDPRM(3) = 0D0
      NLDPRM(4) = 1.420D9

*     Coordinate increments.
      NLDPRM(5) = 360D0/(AXEN(1)-1)
      NLDPRM(6) = 4D6

*     Rest frequency.
      NLDPRM(7) = 1.420D9

*     Annotation.
      IDENTS(1) = 'galactic longitude'
      IDENTS(2) = 'velocity (m/s)'
      IDENTS(3) = 'HI line'

      OPT(1) = 'F'
      OPT(2) = ' '

*     Normal size lettering.
      CALL PGSCH (1.0*SCL)

*     Do tick marks for longitude and grid lines for velocity in yellow.
*     Defer labelling.
      CALL PGSCI (2)
      GCODE(1) = 1
      GCODE(2) = 2
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 2D0, 0, 0D0,
     :   0, 0D0, .FALSE., LNGVEL, 1, 1, 7, ' ', 0, NLDPRM, 256, IC,
     :   CACHE, IERR)

*     Draw fiducial grid lines in white and do labelling.
      CALL PGSCI (1)
      GCODE(1) = 2
      GCODE(2) = 2
      GRID1(1) = 180D0
      GRID2(1) = 0D0
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 0, 0, C0, GCODE, 0D0, 1,
     :   GRID1, 1, GRID2, .FALSE., LNGVEL, 1, 1, 7, ' ', 0, NLDPRM, 256,
     :   IC, CACHE, IERR)

*     Draw the frame.
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)


      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Second example: azimuth-frequency scan.  This sort of output
*     might be obtained from an antenna which scans in azimuth with a
*     receiver which scans simultaneously in frequency.  PGSBOX uses
*     subroutine FSCAN.  Features:
*        - Non-separable (i.e. non-orthogonal) non-linear coordinate
*          system.
*        - Automatic choice of what edges to label; results in labelling
*          the bottom, left and right sides of the plot.
*        - Cyclic labelling.  FSCAN returns the azimuth in the range
*          0 - 720 degrees but PGSBOX is set to normalize this to two
*          cycles of 0 - 360 degrees.
*        - Logarithmic labelling.
*        - Automatic choice of coordinate increments but with request
*          for all grid lines for the logarithmic coordinate.
*        - Degree labelling.
*        - Suppression of common zero arcmin and arcsec fields in
*          sexagesimal degree format.

*     Reference pixel coordinates.
      NLDPRM(1) = 0.5D0
      NLDPRM(2) = 0.5D0

*     Reference pixel values.
      NLDPRM(3) = 0D0
      NLDPRM(4) = 8.5D0

*     Coordinate increments.
      NLDPRM(5) = 720D0/(AXEN(1)+1)
      NLDPRM(6) = 0.002D0

*     Rate of change of NLDPRM(4) with x-pixel.
      NLDPRM(7) = -0.002D0

*     Annotation.
      IDENTS(1) = 'azimuth'
      IDENTS(2) = ESCAPE // 'gn/Hz'
      IDENTS(3) = 'Frequency/azimuth scan'

      OPT(1) = 'D'
      OPT(2) = 'L'

*     Draw full grid lines.
      GCODE(1) = 2
      GCODE(2) = 2

*     Normal size lettering.
      CALL PGSCH (1.0*SCL)

*     Setting LABDEN = 9900 forces all logarithmic grid lines to be
*     drawn.
      CALL PGSCI (1)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 0, 9900, C0, GCODE, 2D0, 0,
     :   0D0, 0, 0D0, .FALSE., FSCAN, 1, 1, 7, ' ', 0, NLDPRM, 256, IC,
     :   CACHE, IERR)

*     Draw the frame.
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Third example: Z versus time plot.  PGSBOX uses subroutine PGCRFN.
*     Features:
*        - Separable (i.e. orthogonal), non-linear coordinate system.
*        - Use of function PGCRFN for separable axis types.
*        - Automatic choice of what edges to label; results in labelling
*          the bottom and left sides of the plot.
*        - Automatic choice of coordinate increments.
*        - Logarithmic labelling over many orders of magnitude.
*        - Single-character annotation on a vertical axis is upright.

*     Function types.
      FCODE(1) = 'Lin '
      FCODE(2) = 'Log '

*     Reference pixel coordinates.
      NLDPRM(1) = 0.5D0
      NLDPRM(2) = -50D0

*     Coordinate increments.
      NLDPRM(3) = 0.04D0
      NLDPRM(4) = 0.02D0

*     Reference pixel values.
      NLDPRM(5) = -3.0D0
      NLDPRM(6) =  1.0D0

*     Annotation.
      IDENTS(1) = 'Age of universe (sec)'
      IDENTS(2) = 'Y'
      IDENTS(3) = ''

      OPT(1) = 'L'
      OPT(2) = ' '

*     Draw ticks for the first coordinate, grid lines for the second.
      GCODE(1) = 1
      GCODE(2) = 2

*     Normal size lettering.
      CALL PGSCH (1.0*SCL)

      CALL PGSCI (1)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 0, 0, C0, GCODE, 2D0, 0, 0D0,
     :   0, 0D0, .FALSE., PGCRFN, 8, 2, 4, FCODE, NLIPRM, NLDPRM, 256,
     :   IC, CACHE, IERR)

*     Draw the frame.
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Fourth example: A simple SIN projection near the south celestial
*     pole.  PGSBOX uses subroutine PGWCSL to interface to WCSLIB.
*     Features:
*        - Non-separable (i.e. non-orthogonal) curvilinear coordinate
*          system.
*        - Use of equivalences to define parameters for PGWCSL.
*        - Discovery of grid lines which do not cross any axis.
*        - Automatic choice of what edges to label; results in labelling
*          all sides of the plot.
*        - Automatic choice of coordinate increments but with request
*          for increased grid density for each coordinate.
*        - Double precision accuracy.
*        - Cyclic coordinates.  PGWCSL returns the right ascension in
*          the range -180 to +180 degrees, i.e. with a discontinuity
*          at +/- 180 degrees.
*        - Labelling of degrees as time in the range 0 - 24h.
*        - Suppression of labels which would overlap one another.
*        - Sexagesimal degree labelling with automatically determined
*          precision.
*        - Suppression of common zero minute and second fields in
*          sexagesimal time format.

*     Set projection type to SIN.
      CTYPE(1) = 'RA---SIN'
      CTYPE(2) = 'DEC--SIN'

*     Reference pixel coordinates (EQUIVALENCE'd into NLDPRM above).
      CRPIX(1) = 384D0
      CRPIX(2) = 256D0

*     Linear transformation matrix (EQUIVALENCE'd into NLDPRM).
      PC(1,1) = 1D0
      PC(1,2) = 0D0
      PC(2,1) = 0D0
      PC(2,2) = 1D0

*     Coordinate increments (EQUIVALENCE'd into NLDPRM).
      CDELT(1) = -1D0/3600000D0
      CDELT(2) =  1D0/3600000D0

*     Spherical coordinate references (EQUIVALENCE'd into NLDPRM).
      CRVAL(1) =  25.000D0
      CRVAL(2) = -89.99995D0
      LNGPOL = 999D0
      LATPOL = 999D0

*     Set parameters for an NCP projection (EQUIVALENCE'd into NLDPRM).
      PROJP(1)  = 0D0
      PROJP(2)  = COSD(CRVAL(2))/SIND(CRVAL(2))

*     Annotation.
      IDENTS(1) = 'Right ascension'
      IDENTS(2) = 'Declination'
      IDENTS(3) = 'WCS SIN projection'

      OPT(1) = 'G'
      OPT(2) = 'E'

*     Draw full grid lines.
      GCODE(1) = 2
      GCODE(2) = 2

*     Compact lettering.
      CALL PGSCH (0.8*SCL)

*     Draw the celestial grid.  The grid density is set for each world
*     coordinate by specifying LABDEN = 1224.
      CALL PGSCI (1)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 0, 1224, C0, GCODE, 0D0, 0,
     :   0D0, 0, 0D0, .FALSE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the frame.
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Fifth example: Conic equal area.  PGSBOX uses subroutine PGWCSL
*     to interface to WCSLIB.  Features:
*        - Non-separable (i.e. non-orthogonal) curvilinear coordinate
*          system.
*        - Coordinate system undefined in areas of the plot.
*        - Use of equivalences to define parameters for PGWCSL.
*        - Discontinuous grid lines handled by PGWCSL.
*        - Discovery of grid lines which do not cross any axis.
*        - Colour control for grid and labelling.
*        - Reduced size lettering.
*        - Automatic choice of what edges to label; results in labelling
*          all sides of the plot.
*        - Automatic choice of coordinate increments.
*        - Cyclic coordinates.  PGWCSL returns the longitude in the
*          range -180 to +180 degrees, i.e. with a discontinuity at
*          +/- 180 degrees.
*        - Suppression of labels which would overlap one another.
*        - Suppression of common zero arcmin and arcsec fields in
*          sexagesimal degree format.

*     Set projection type to conic equal-area.
      CTYPE(1) = 'RA---COE'
      CTYPE(2) = 'DEC--COE'

*     Reference pixel coordinates (EQUIVALENCE'd into NLDPRM above).
      CRPIX(1) = 256D0
      CRPIX(2) = 256D0

*     Linear transformation matrix (EQUIVALENCE'd into NLDPRM).
      PC(1,1) = 1D0
      PC(1,2) = 0D0
      PC(2,1) = 0D0
      PC(2,2) = 1D0

*     Coordinate increments (EQUIVALENCE'd into NLDPRM).
      CDELT(1) = -1D0/3D0
      CDELT(2) =  1D0/3D0

*     Middle latitude and offset from standard parallels.
      PROJP(1)  = 60D0
      PROJP(2)  = 15D0

*     Spherical coordinate references (EQUIVALENCE'd into NLDPRM).
      CRVAL(1) = 90D0
      CRVAL(2) = 30D0
      LNGPOL = 150D0
      LATPOL = 999D0

*     Annotation.
      IDENTS(1) = 'longitude'
      IDENTS(2) = 'latitude'
      IDENTS(3) = 'WCS conic equal area projection'

      OPT(1) = 'E'
      OPT(2) = 'E'

*     Draw full grid lines.
      GCODE(1) = 2
      GCODE(2) = 2

*     Reduced size lettering.
      CALL PGSCH (0.8*SCL)

*     Use colour to associate grid lines and labels.
*     Meridians in red.
      CALL PGSCR (10, 0.5, 0.0, 0.0)
*     Parallels in blue.
      CALL PGSCR (11, 0.0, 0.2, 0.5)
*     Longitudes in red.
      CALL PGSCR (12, 0.8, 0.3, 0.0)
*     Latitudes in blue.
      CALL PGSCR (13, 0.0, 0.4, 0.7)
*     Longitude labels in red.
      CALL PGSCR (14, 0.8, 0.3, 0.0)
*     Latitude labels in blue.
      CALL PGSCR (15, 0.0, 0.4, 0.7)
*     Title in cyan.
      CALL PGSCR (16, 0.3, 1.0, 1.0)
 
      CI(1) = 10
      CI(2) = 11
      CI(3) = 12
      CI(4) = 13
      CI(5) = 14
      CI(6) = 15
      CI(7) = 16

*     Draw the celestial grid letting PGSBOX choose the increments.
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 0, 0, CI, GCODE, 0D0, 0, 0D0,
     :   0, 0D0, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM, 256, IC,
     :   CACHE, IERR)

*     Set parameters to draw the native grid.
      CRVAL(1) = 0D0
      CRVAL(2) = PROJP(1)
      LNGPOL = 999D0
      LATPOL = 999D0

*     We just want to delineate the boundary, in green.
      GRID1(1) = -180D0
      GRID1(2) =  180D0
      GRID2(1) =  -90D0
      GRID2(2) =   90D0
      CALL PGSCI (7)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 2,
     :   GRID1, 2, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the frame.
      CALL PGSCI (1)
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Sixth example: Polyconic with colour-coded grid.  PGSBOX uses
*     subroutine PGWCSL to interface to WCSLIB.  Features:
*        - Non-separable (i.e. non-orthogonal) curvilinear coordinate
*          system.
*        - Coordinate system undefined in areas of the plot.
*        - Use of equivalences to define parameters for PGWCSL.
*        - Discontinuous grid lines handled by PGWCSL.
*        - Colour coded labelling.
*        - Colour coded grid implemented by the caller.
*        - Basic management of the axis-crossing table (see code).
*        - Reduced size lettering.
*        - Tick marks external to the frame.
*        - User selection of what edges to label with request for both
*          coordinates to be labelled on bottom, left and top edges.
*        - User selection of which grid lines to plot.
*        - Concatenation of annotation at bottom and left; automatically
*          suppressed at the top since only one coordinate is labelled
*          there.
*        - Suppression of labels which would overlap one another.
*        - Degree labelling.
*        - Labelling of degrees as time in the range -12 - +12h.
*        - Suppression of common zero minute and second fields in
*          sexagesimal time format.

*     Set projection type to polyconic.
      CTYPE(1) = 'RA---PCO'
      CTYPE(2) = 'DEC--PCO'

*     Reference pixel coordinates (EQUIVALENCE'd into NLDPRM above).
      CRPIX(1) = 192D0
      CRPIX(2) = 640D0

*     Rotate 30 degrees.
      PC(1,1) =  COSD(30D0)
      PC(1,2) =  SIND(30D0)
      PC(2,1) = -SIND(30D0)
      PC(2,2) =  COSD(30D0)

*     Coordinate increments (EQUIVALENCE'd into NLDPRM).
      CDELT(1) = -1D0/5D0
      CDELT(2) =  1D0/5D0

*     Spherical coordinate references (EQUIVALENCE'd into NLDPRM).
      CRVAL(1) = 332D0
      CRVAL(2) =  40D0
      LNGPOL = -30D0
      LATPOL = 999D0

*     Annotation.
      IDENTS(1) = 'Hour angle'
      IDENTS(2) = 'Declination'
      IDENTS(3) = 'WCS polyconic projection'

      OPT(1) = 'H'
      OPT(2) = 'B'

*     Reduced size lettering.
      CALL PGSCH (0.9*SCL)

*     Draw external (TIKLEN < 0) tick marks every 5 degrees.
      GCODE(1) = 1
      GCODE(2) = 1
      TIKLEN = -2D0
      CALL PGSCI (6)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, TIKLEN, 0,
     :   5D0, 0, 5D0, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Resetting the table index to zero causes information about the
*     tick marks to be discarded.
      IC = 0

*     Now reset to draw full grid lines rather than tick marks.
      GCODE(1) = 2
      GCODE(2) = 2

*     Draw the primary meridian and equator in yellow.
      GRID1(1) = 0D0
      GRID2(1) = 0D0
      CALL PGSCI (2)
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 1,
     :   GRID1, 1, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     At this point the axis-crossing table will have entries for the
*     primary meridian and equator.  Labelling was deferred in the
*     previous call, and the table is passed intact on the second call
*     to accumulate further axis-crossings.

*     Draw 90 degree meridians and poles in white.
      GRID1(1) =  90D0
      GRID1(2) = 180D0
      GRID1(3) = 270D0
      GRID2(1) = -90D0
      GRID2(2) =  90D0
      CALL PGSCI (3)
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 3,
     :   GRID1, 2, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the first set of 15 degree meridians and parallels in blue.
      GRID1(1) =  15D0
      GRID1(2) =  60D0
      GRID1(3) = 105D0
      GRID1(4) = 150D0
      GRID1(5) = 195D0
      GRID1(6) = 240D0
      GRID1(7) = 285D0
      GRID1(8) = 330D0
      GRID2(1) = -75D0
      GRID2(2) = -30D0
      GRID2(3) =  15D0
      GRID2(4) =  60D0
      CALL PGSCI (4)
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 8,
     :   GRID1, 4, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the second set of 15 degree meridians and parallels in red.
      GRID1(1) =  30D0
      GRID1(2) =  75D0
      GRID1(3) = 120D0
      GRID1(4) = 165D0
      GRID1(5) = 210D0
      GRID1(6) = 255D0
      GRID1(7) = 300D0
      GRID1(8) = 345D0
      GRID2(1) = -60D0
      GRID2(2) = -15D0
      GRID2(3) =  30D0
      GRID2(4) =  75D0
      CALL PGSCI (5)
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 8,
     :   GRID1, 4, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     The axis-crossing table has now accumulated information for all of
*     the preceding meridians and parallels but no labels have been
*     produced.  It will acquire information for the the next set of
*     meridians and parallels before being processed by this call to
*     PGSBOX which finally produces the labels.

*     Draw the 45 degree meridians and parallels in grey and use colour
*     to differentiate grid labels.
*     Meridians and parallels in grey.
      CALL PGSCR (10, 0.7, 0.7, 0.7)
      CALL PGSCR (11, 0.7, 0.7, 0.7)
*     Longitudes tinged red.
      CALL PGSCR (12, 1.0, 0.9, 0.6)
*     Latitudes tinged green.
      CALL PGSCR (13, 0.8, 1.0, 0.9)
*     Longitude labels tinged red.
      CALL PGSCR (14, 1.0, 0.9, 0.6)
*     Latitude labels tinged green.
      CALL PGSCR (15, 0.8, 1.0, 0.9)
*     Title in white.
      CALL PGSCR (16, 1.0, 1.0, 1.0)

      CI(1) = 10
      CI(2) = 11
      CI(3) = 12
      CI(4) = 13
      CI(5) = 14
      CI(6) = 15
      CI(7) = 16

*     Tell PGSBOX what edges to label.
      GRID1(1) =  45D0
      GRID1(2) = 135D0
      GRID1(3) = 225D0
      GRID1(4) = 315D0
      GRID2(1) = -45D0
      GRID2(2) =  45D0
      CALL PGSCI (6)
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 2333, 0, CI, GCODE, 0D0, 4,
     :   GRID1, 2, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Native grid in green (delineates boundary).
      GRID1(1) = -180D0
      GRID1(2) =  180D0
      GRID2(1) = -999D0
      CRVAL(1) = 0D0
      CRVAL(2) = 0D0
      LNGPOL = 999D0
      LATPOL = 999D0
      CALL PGSCI (7)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, -1, 0, C0, GCODE, 0D0, 2,
     :   GRID1, 1, GRID2, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the frame.
      CALL PGSCI (1)
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Seventh example: Plate Carree projection.  PGSBOX uses subroutine
*     PGWCSL to interface to WCSLIB.  Features:
*        - Rectangular image.
*        - Dual coordinate grids.
*        - Non-separable (i.e. non-orthogonal) curvilinear coordinate
*          system.
*        - Use of equivalences to define parameters for PGWCSL.
*        - Discontinuous grid lines handled by PGWCSL.
*        - Colour coding of grid and labelling.
*        - Reduced size lettering.
*        - Manual labelling control.
*        - Manual and automatic choice of coordinate increments.
*        - Cyclic coordinates.  PGWCSL returns the longitude in the
*          range -180 to +180 degrees, i.e. with a discontinuity at
*          +/- 180 degrees.
*        - Suppression of labels which would overlap one another.

      AXEN(1) = 181
      AXEN(2) =  91

      BLC(1) = 0.5
      BLC(2) = 0.5
      TRC(1) = AXEN(1) + 0.5
      TRC(2) = AXEN(2) + 0.5

*     Reset viewport for rectangular image.

      IF (LARGE) THEN
         CALL PGVSTD ()
      ELSE
         CALL PGVSIZ (1.0, 3.0, 1.0, 3.0)
      END IF
      CALL PGWNAD (0.0, 1.0, 0.0, REAL(AXEN(2))/REAL(AXEN(1)))

*     Set projection type to plate carree.
      CTYPE(1) = 'GLON-CAR'
      CTYPE(2) = 'GLAT-CAR'

*     Reference pixel coordinates (EQUIVALENCE'd into NLDPRM above).
      CRPIX(1) = 226.0D0
      CRPIX(2) =  46.0D0

      crpix(1) =  crpix(1) - 180d0

*     Linear transformation matrix (EQUIVALENCE'd into NLDPRM).
      PC(1,1) = 1D0
      PC(1,2) = 0D0
      PC(2,1) = 0D0
      PC(2,2) = 1D0

*     Coordinate increments (EQUIVALENCE'd into NLDPRM).
      CDELT(1) = -1D0
      CDELT(2) =  1D0

*     Set parameters to draw the native grid.
      CRVAL(1) = 0D0
      CRVAL(2) = 0D0
      LNGPOL = 999D0
      LATPOL = 999D0

      crval(1) =  crval(1) + 180d0
      crval(2) = -crval(2)

*     Annotation.
      IDENTS(1) = ''
      IDENTS(2) = ''
      IDENTS(3) = 'WCS plate caree projection'

      OPT(1) = 'C'
      OPT(2) = 'C'

*     Draw full grid lines.
      GCODE(1) = 2
      GCODE(2) = 2

*     Reduced size lettering.
      CALL PGSCH (0.8*SCL)

*     Draw native grid in green.
      CALL PGSCR (16, 0.0, 0.2, 0.0)
*     Title in cyan.
      CALL PGSCR (17, 0.3, 1.0, 1.0)

      CI(1) = 16
      CI(2) = 16
      CI(3) =  7
      CI(4) =  7
      CI(5) = -1
      CI(6) = -1
      CI(7) = 17

      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 2100, 0, CI, GCODE, 0D0, 0,
     :   15D0, 0, 15D0, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM,
     :   256, IC, CACHE, IERR)


*     Spherical coordinate references (EQUIVALENCE'd into NLDPRM).
      CRVAL(1) = 30D0
      CRVAL(2) = 35D0
      LNGPOL = 999D0
      LATPOL = 999D0

      crval(1) =  crval(1) + 180d0
      crval(2) = -crval(2)

*     Annotation.
      IDENTS(1) = 'longitude'
      IDENTS(2) = 'latitude'
      IDENTS(3) = ''

      OPT(1) = 'E'
      OPT(2) = 'E'

*     Use colour to associate grid lines and labels.
*     Meridians in red.
      CALL PGSCR (10, 0.5, 0.0, 0.0)
*     Parallels in blue.
      CALL PGSCR (11, 0.0, 0.2, 0.5)
*     Longitudes in red.
      CALL PGSCR (12, 0.8, 0.3, 0.0)
*     Latitudes in blue.
      CALL PGSCR (13, 0.0, 0.4, 0.7)
*     Longitude labels in red.
      CALL PGSCR (14, 0.8, 0.3, 0.0)
*     Latitude labels in blue.
      CALL PGSCR (15, 0.0, 0.4, 0.7)

      CI(1) = 10
      CI(2) = 11
      CI(3) = 12
      CI(4) = 13
      CI(5) = 14
      CI(6) = 15
      CI(7) = -1

*     Draw the celestial grid letting PGSBOX choose the increments.
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 21, 0, CI, GCODE, 0D0, 0, 0D0,
     :   0, 0D0, .TRUE., PGWCSL, 16, 4, 50, CTYPE, WCS, NLDPRM, 256, IC,
     :   CACHE, IERR)

*     Draw the frame.
      CALL PGSCI (1)
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

      CALL PGPAGE ()

*-----------------------------------------------------------------------

*     Eighth example: linear-linear plot with two types of alternative
*     labelling.  PGSBOX uses subroutine PGCRFN.  Features:
*        - Separable (i.e. orthogonal), linear coordinate system.
*        - Use of function PGCRFN for separable axis types.
*        - Alternative labelling and axis annotation.
*        - Direct manipulation of the axis-crossing table.
*        - Tick mark and grid line control.
*        - User selection of what edges to label.
*        - Automatic choice of coordinate increments.

      IF (LARGE) THEN
         CALL PGVSTD ()
      ELSE
         CALL PGVSIZ (1.0, 3.0, 1.0, 3.0)
      END IF
      CALL PGWNAD (0.0, 1.0, 0.0, 1.0)

      AXEN(1) = 512
      AXEN(2) = 512

      BLC(1) = 0.5
      BLC(2) = 0.5
      TRC(1) = AXEN(1) + 0.5
      TRC(2) = AXEN(2) + 0.5

*     Function types.
      FCODE(1) = 'Lin '
      FCODE(2) = 'Lin '

*     Reference pixel coordinates.
      NLDPRM(1) = 0.5D0
      NLDPRM(2) = 0.5D0

*     Coordinate increments.
      NLDPRM(3) = 0.03D0
      NLDPRM(4) = 0.03D0

*     Reference pixel values.
      NLDPRM(5) = 20D0
      NLDPRM(6) =  0D0

*     Annotation.
      IDENTS(1) = 'temperature of frog (' // ESCAPE // 'uo' //
     :            ESCAPE // 'dC)'
      IDENTS(2) = 'distance hopped (m)'
      IDENTS(3) = ''

      OPT(1) = ' '
      OPT(2) = ' '

*     Draw tick marks at the bottom for the first coordinate, grid lines
*     for the second.  Setting GCODE(1) = -1 inhibits information being
*     stored for labels on the top edge while GCODE(2) = 2 causes
*     information to be stored for labels on the right edge even if
*     those labels are not actually produced.
      GCODE(1) = -1
      GCODE(2) = 2

*     Normal size lettering.
      CALL PGSCH (1.0*SCL)

*     Set LABCTL = 21 to label the bottom and left edges only.
      CALL PGSCI (1)
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 21, 0, C0, GCODE, 2D0, 0, 0D0,
     :   0, 0D0, .FALSE., PGCRFN, 8, 2, 4, FCODE, NLIPRM, NLDPRM, 256,
     :   IC, CACHE, IERR)

*     Information for labels on the right edge was stored in the
*     crossing table on the first call to PGSBOX.  We now want to
*     manipulate it to convert metres to feet.  Note that while it's
*     a simple matter to draw alternative sets of tick marks on opposite
*     edges of the frame, as with the two temperature scales, we have
*     the slightly more difficult requirement of labelling grid lines
*     with different values at each end.
      DO 10 J = 1, IC
*        Look for entries associated with the right edge of the frame.
         IF (CACHE(1,J).EQ.4D0) THEN
*           Convert to feet, rounding to the nearest 0.1.
            CACHE(4,J) = CACHE(4,J) * 1D3/(25.4*12D0)
            CACHE(4,J) = AINT(CACHE(4,J)*10D0 + 0.5D0)/10D0
         END IF
 10   CONTINUE

*     Annotation for the right edge.
      IDENTS(1) = ''
      IDENTS(2) = '(feet)'

*     Set LABCTL = 12000 to label the right edge with the second
*     coordinate without redrawing the grid lines.
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 12000, 0, C0, GCODE, 2D0, 0,
     :   0D0, 0, 0D0, .FALSE., PGCRFN, 8, 2, 4, FCODE, NLIPRM, NLDPRM,
     :   256, IC, CACHE, IERR)

*     The alternative temperature scale in Fahrenheit is to be
*     constructed with a new set of tick marks.
      NLDPRM(3) = NLDPRM(3)*1.8D0
      NLDPRM(5) = NLDPRM(5)*1.8D0 + 32D0
 
*     Draw tick marks at the top for the first coordinate, don't redo
*     grid lines for the second.
      GCODE(1) = -100
      GCODE(2) = 0

*     Annotation for the top edge.
      IDENTS(1) = '(' // ESCAPE // 'uo' // ESCAPE // 'dF)'
      IDENTS(2) = ''
 
*     Set LABCTL = 100 to label the top edge; Set IC = -1 to redetermine
*     the coordinate extrema.
      IC = -1
      CALL PGSBOX (BLC, TRC, IDENTS, OPT, 100, 0, C0, GCODE, 2D0, 0,
     :   0D0, 0, 0D0, .FALSE., PGCRFN, 8, 2, 4, FCODE, NLIPRM, NLDPRM,
     :   256, IC, CACHE, IERR)

*     Draw the frame.
      CALL PGBOX ('BC', 0.0, 0, 'BC', 0.0, 0)

*-----------------------------------------------------------------------

      CALL PGEND ()


      END
