*=======================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2002, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or
*   modify it under the terms of the GNU Library General Public
*   License as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This library is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   Library General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public
*   License along with this library; if not, write to the Free
*   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=======================================================================
*
*   FORTRAN routines for the spherical coordinate transformations used
*   by the FITS "World Coordinate System" (WCS) convention.
*
*   Summary of routines
*   -------------------
*   The spherical coordinate transformations are implemented via
*   separate subroutines for the transformation in each direction.
*
*   Forward transformation; SPHFWD
*   ------------------------------
*   Transform celestial coordinates to the native coordinates of a
*   projection.
*
*   Given:
*      LNG,LAT  D        Celestial longitude and latitude, in degrees.
*      EUL      D(5)     Euler angles for the transformation:
*                          1: Celestial longitude of the native pole, in
*                             degrees.
*                          2: Celestial colatitude of the native pole,
*                             or native colatitude of the celestial
*                             pole, in degrees.
*                          3: Native longitude of the celestial pole, in
*                             degrees.
*                          4: cos(EUL(2))
*                          5: sin(EUL(2))
*
*   Returned:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*      IERR     I        Error status; 0 means success.
*
*   Reverse transformation; SPHREV
*   ------------------------------
*   Transform native coordinates of a projection to celestial
*   coordinates.
*
*   Given:
*      PHI,     D        Longitude and latitude in the native coordinate
*      THETA             system of the projection, in degrees.
*      EUL      D(5)     Euler angles for the transformation:
*                          1: Celestial longitude of the native pole, in
*                             degrees.
*                          2: Celestial colatitude of the native pole,
*                             or native colatitude of the celestial
*                             pole, in degrees.
*                          3: Native longitude of the celestial pole, in
*                             degrees.
*                          4: cos(EUL(2))
*                          5: sin(EUL(2))
*
*   Returned:
*      LNG,LAT  D        Celestial longitude and latitude, in degrees.
*      IERR     I        Error status; 0 means success.
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: sph.f,v 2.7 2002/04/03 01:24:44 mcalabre Exp $
*=======================================================================
      SUBROUTINE SPHFWD (LNG, LAT, EUL, PHI, THETA, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COSLAT, COSLNG, DLNG, DPHI, EUL(5), LAT, LNG,
     *          PHI, SINLAT, SINLNG, THETA, TOL, X, Y, Z

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-5)

      DOUBLE PRECISION ACOSD, ASIND, ATAN2D, COSD, SIND
*-----------------------------------------------------------------------
      COSLAT = COSD(LAT)
      SINLAT = SIND(LAT)

      DLNG = LNG - EUL(1)
      COSLNG = COSD(DLNG)
      SINLNG = SIND(DLNG)

*     Compute the native longitude.
      X = SINLAT*EUL(5) - COSLAT*EUL(4)*COSLNG
      IF (ABS(X).LT.TOL) THEN
*        Rearrange formula to reduce roundoff errors.
         X = -COSD(LAT+EUL(2)) + COSLAT*EUL(4)*(1D0 - COSLNG)
      END IF
      Y = -COSLAT*SINLNG
      IF (X.NE.0D0 .OR. Y.NE.0D0) THEN
         DPHI = ATAN2D(Y, X)
      ELSE
*        Change of origin of longitude.
         DPHI = DLNG - 180D0
      END IF
      PHI = EUL(3) + DPHI

*     Normalize the native longitude.
      IF (PHI.GT.180D0) THEN
         PHI = PHI - 360D0
      ELSE IF (PHI.LT.-180D0) THEN
         PHI = PHI + 360D0
      END IF

*     Compute the native latitude.
      IF (MOD(DLNG,180D0).EQ.0D0) THEN
         THETA = LAT + COSLNG*EUL(2)
         IF (THETA.GT.90D0)  THETA =  180D0 - THETA
         IF (THETA.LT.-90D0) THETA = -180D0 - THETA
      ELSE
         Z = SINLAT*EUL(4) + COSLAT*EUL(5)*COSLNG
         IF (ABS(Z).GT.0.99D0) THEN
*           Use an alternative formula for greater numerical accuracy.
            THETA = SIGN(ACOSD(SQRT(X*X+Y*Y)), Z)
         ELSE
            THETA = ASIND(Z)
         END IF
      END IF

      IERR = 0
      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE SPHREV (PHI, THETA, EUL, LNG, LAT, IERR)
*-----------------------------------------------------------------------
      INTEGER   IERR
      DOUBLE PRECISION COSPHI, COSTHE, DLNG, DPHI, EUL(5), LAT, LNG,
     *          PHI, SINPHI, SINTHE, THETA, TOL, X, Y, Z

      DOUBLE PRECISION D2R, PI, R2D
      PARAMETER (PI = 3.141592653589793238462643D0)
      PARAMETER (D2R = PI/180D0, R2D = 180D0/PI)

      PARAMETER (TOL = 1D-5)

      DOUBLE PRECISION ACOSD, ASIND, ATAN2D, COSD, SIND
*-----------------------------------------------------------------------
      COSTHE = COSD(THETA)
      SINTHE = SIND(THETA)

      DPHI = PHI - EUL(3)
      COSPHI = COSD(DPHI)
      SINPHI = SIND(DPHI)

*     Compute the celestial longitude.
      X = SINTHE*EUL(5) - COSTHE*EUL(4)*COSPHI
      IF (ABS(X).LT.TOL) THEN
*        Rearrange formula to reduce roundoff errors.
         X = -COSD(THETA+EUL(2)) + COSTHE*EUL(4)*(1D0 - COSPHI)
      END IF
      Y = -COSTHE*SINPHI
      IF (X.NE.0D0 .OR. Y.NE.0D0) THEN
         DLNG = ATAN2D(Y, X)
      ELSE
*        Change of origin of longitude.
         DLNG = DPHI + 180D0
      END IF
      LNG = EUL(1) + DLNG

*     Normalize the celestial longitude.
      IF (EUL(1).GE.0D0) THEN
         IF (LNG.LT.0D0) LNG = LNG + 360D0
      ELSE
         IF (LNG.GT.0D0) LNG = LNG - 360D0
      END IF

      IF (LNG.GT.360D0) THEN
         LNG = LNG - 360D0
      ELSE IF (LNG.LT.-360D0) THEN
         LNG = LNG + 360D0
      END IF

*     Compute the celestial latitude.
      IF (MOD(DPHI,180D0).EQ.0D0) THEN
         LAT = THETA + COSPHI*EUL(2)
         IF (LAT.GT.90D0)  LAT =  180D0 - LAT
         IF (LAT.LT.-90D0) LAT = -180D0 - LAT
      ELSE
         Z = SINTHE*EUL(4) + COSTHE*EUL(5)*COSPHI
         IF (ABS(Z).GT.0.99D0) THEN
*           Use an alternative formula for greater numerical accuracy.
            LAT = SIGN(ACOSD(SQRT(X*X+Y*Y)), Z)
         ELSE
            LAT = ASIND(Z)
         END IF
      END IF


      IERR = 0
      RETURN
      END
