/*============================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2002, Mark Calabretta
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation; either version 2 of the License, or (at your
*   option) any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You should have received a copy of the GNU General Public License along
*   with this library; if not, write to the Free Software Foundation, Inc.,
*   675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=============================================================================
*
*   tspec tests the spectral transformation routines for closure.
*
*   $Id$
*---------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <spec.h>

#ifndef __STDC__
#ifndef const
#define const
#endif
#endif

const int nspec = 9991;
const double tol = 1.0e-9;

void closure (const char *, const char *, double, int (*)(), int (*)(),
              const double [], double []);

const double C = 2.99792458e8;


int main()

{
   int    stat[nspec], status;
   double restfrq, restwav, step, x;
   double awav[nspec], freq[nspec], spc1[nspec], spc2[nspec], velo[nspec],
          wave[nspec];
   struct specvars specs;

   register int j, k;

   /* Uncomment the following two lines to raise SIGFPE on floating point
    * exceptions for the Sun FORTRAN compiler.  This signal can be caught
    * within 'dbx' by issuing the command "catch FPE".
    */
/* #include <floatingpoint.h> */
/* ieee_handler("set", "common", SIGFPE_ABORT); */


   printf("\nTesting closure of WCSLIB spectral transformation routines\n");
   printf("----------------------------------------------------------\n");

   /* List error messages. */
   printf("\nList of spec error codes:\n");
   for (j = 1; j <= 2 ; j++) {
      printf("   %d: %s.\n", j, spec_errmsg[j]);
   }

   restfrq = 1420.40595e6;
   restwav = C/restfrq;


   /* Exercise specx(). */
   printf("\nTesting spectral cross-conversions (specx).\n\n");
   if (status = specx("VELO", 4.3e5, restfrq, restwav, &specs)) {
      printf("   specx: %s.\n", spec_errmsg[status]);
      return 1;
   }

   printf("    restfrq: %20.12E\n", specs.restfrq);
   printf("    restwav: %20.12E\n", specs.restwav);
   printf("\n");
   printf("       freq: %20.12E\n", specs.freq);
   printf("       afrq: %20.12E\n", specs.afrq);
   printf("       ener: %20.12E\n", specs.ener);
   printf("       wavn: %20.12E\n", specs.wavn);
   printf("       vrad: %20.12E\n", specs.vrad);
   printf("       wave: %20.12E\n", specs.wave);
   printf("       vopt: %20.12E\n", specs.vopt);
   printf("       zopt: %20.12E\n", specs.zopt);
   printf("       awav: %20.12E\n", specs.awav);
   printf("       velo: %20.12E\n", specs.velo);
   printf("       beta: %20.12E\n", specs.beta);
   printf("\n");

   printf("dfreq/dafrq: %20.12E\n", specs.dfreqafrq);
   printf("dafrq/dfreq: %20.12E\n", specs.dafrqfreq);

   printf("dfreq/dener: %20.12E\n", specs.dfreqener);
   printf("dener/dfreq: %20.12E\n", specs.denerfreq);

   printf("dfreq/dwavn: %20.12E\n", specs.dfreqwavn);
   printf("dwavn/dfreq: %20.12E\n", specs.dwavnfreq);

   printf("dfreq/dvrad: %20.12E\n", specs.dfreqvrad);
   printf("dvrad/dfreq: %20.12E\n", specs.dvradfreq);

   printf("dfreq/dwave: %20.12E\n", specs.dfreqwave);
   printf("dwave/dfreq: %20.12E\n", specs.dwavefreq);

   printf("dfreq/dawav: %20.12E\n", specs.dfreqawav);
   printf("dawav/dfreq: %20.12E\n", specs.dawavfreq);

   printf("dfreq/dvelo: %20.12E\n", specs.dfreqvelo);
   printf("dvelo/dfreq: %20.12E\n", specs.dvelofreq);

   printf("dwave/dvopt: %20.12E\n", specs.dwavevopt);
   printf("dvopt/dwave: %20.12E\n", specs.dvoptwave);

   printf("dwave/dzopt: %20.12E\n", specs.dwavezopt);
   printf("dzopt/dwave: %20.12E\n", specs.dzoptwave);

   printf("dwave/dawav: %20.12E\n", specs.dwaveawav);
   printf("dawav/dwave: %20.12E\n", specs.dawavwave);

   printf("dwave/dvelo: %20.12E\n", specs.dwavevelo);
   printf("dvelo/dwave: %20.12E\n", specs.dvelowave);

   printf("dawav/dvelo: %20.12E\n", specs.dawavvelo);
   printf("dvelo/dawav: %20.12E\n", specs.dveloawav);

   printf("dvelo/dbeta: %20.12E\n", specs.dvelobeta);
   printf("dbeta/dvelo: %20.12E\n", specs.dbetavelo);
   printf("\n");


   /* Construct a linear velocity spectrum. */
   step = (2.0*C/nspec) / 2.0;
   for (j = 0, k = -nspec; j < nspec; j++, k += 2) {
      velo[j] = (k+1)*step;
   }
   printf("\nVelocity range: %.3f to %.3f km/s, step: %.3f km/s\n",
          velo[0]*1e-3, velo[nspec-1]*1e-3, (velo[1] - velo[0])*1e-3);

   /* Convert it to frequency. */
   velofreq(restfrq, nspec, 1, 1, velo, freq, stat);

   /* Test closure of all two-way combinations. */
   closure("freq", "afrq", 0.0,     freqafrq, afrqfreq, freq, spc1);
   closure("afrq", "freq", 0.0,     afrqfreq, freqafrq, spc1, spc2);

   closure("freq", "ener", 0.0,     freqener, enerfreq, freq, spc1);
   closure("ener", "freq", 0.0,     enerfreq, freqener, spc1, spc2);

   closure("freq", "wavn", 0.0,     freqwavn, wavnfreq, freq, spc1);
   closure("wavn", "freq", 0.0,     wavnfreq, freqwavn, spc1, spc2);

   closure("freq", "vrad", restfrq, freqvrad, vradfreq, freq, spc1);
   closure("vrad", "freq", restfrq, vradfreq, freqvrad, spc1, spc2);

   closure("freq", "wave", 0.0,     freqwave, wavefreq, freq, wave);
   closure("wave", "freq", 0.0,     wavefreq, freqwave, wave, spc2);

   closure("freq", "awav", 0.0,     freqawav, awavfreq, freq, awav);
   closure("awav", "freq", 0.0,     awavfreq, freqawav, awav, spc2);

   closure("freq", "velo", restfrq, freqvelo, velofreq, freq, velo);
   closure("velo", "freq", restfrq, velofreq, freqvelo, velo, spc2);

   closure("wave", "vopt", restwav, wavevopt, voptwave, wave, spc1);
   closure("vopt", "wave", restwav, voptwave, wavevopt, spc1, spc2);

   closure("wave", "zopt", restwav, wavezopt, zoptwave, wave, spc1);
   closure("zopt", "wave", restwav, zoptwave, wavezopt, spc1, spc2);

   closure("wave", "awav", 0.0,     waveawav, awavwave, wave, spc1);
   closure("awav", "wave", 0.0,     awavwave, waveawav, spc1, spc2);

   closure("wave", "velo", restwav, wavevelo, velowave, wave, spc1);
   closure("velo", "wave", restwav, velowave, wavevelo, spc1, spc2);

   closure("awav", "velo", restwav, awavvelo, veloawav, awav, spc1);
   closure("velo", "awav", restwav, veloawav, awavvelo, spc1, spc2);

   closure("velo", "beta", 0.0,     velobeta, betavelo, velo, spc1);
   closure("beta", "velo", 0.0,     betavelo, velobeta, spc1, spc2);


   /* Construct a logarithmic spectrum and test closure. */
   step = (40.0/nspec) / 2.0;
   for (j = 0, k = -nspec; j < nspec; j++, k += 2) {
      spc1[j] = k*step;
   }
   printf("\nLogarithmic range: %.1f to %.1f, step: %.1f\n", spc1[0],
          spc1[nspec-1], spc1[1] - spc1[0]);

   closure("log", "spec", 0.0, logspec, speclog, spc1, spc2);
   closure("spec", "log", 0.0, speclog, logspec, spc2, spc1);

   return 0;
}

/*--------------------------------------------------------------------------*/

void closure (from, to, parm, fwd, rev, spec1, spec2)

const char *from, *to;
double parm;
SPEC_PROTO((*fwd))
SPEC_PROTO((*rev))
const double spec1[];
double spec2[];

{
   static char skip = '\0';
   int stat1[nspec], stat2[nspec], status;
   register int j;
   double clos[nspec], resid, residmax;

   /* Convert the first to the second. */
   if (status = fwd(parm, nspec, 1, 1, spec1, spec2, stat1)) {
      printf("%s%s: %s.\n", from, to, spec_errmsg[status]);
   }

   /* Convert the second back to the first. */
   if (status = rev(parm, nspec, 1, 1, spec2, clos, stat2)) {
      printf("%s%s: %s.\n", to, from, spec_errmsg[status]);
   }

   residmax = 0.0;

   /* Test closure. */
   for (j = 0; j < nspec; j++) {
      if (stat1[j]) {
         printf("%c%s%s: %s = %.12E -> %s = ???, stat = %d\n", skip, from, to,
                from, spec1[j], to, stat1[j]);
         skip = '\0';
         continue;
      }

      if (stat2[j]) {
         printf("%c%s%s: %s = %.12E -> %s = %.12E -> %s = ???, stat = %d\n",
                skip, from, to, from, spec1[j], to, spec2[j], from, stat2[j]);
         skip = '\0';
         continue;
      }

      if (spec1[j] == 0.0) {
         resid = fabs(clos[j] - spec1[j]);
      } else {
         resid = fabs((clos[j] - spec1[j])/spec1[j]);
         if (resid > residmax) residmax = resid;
      }

      if (resid > tol) {
         printf("%c%s%s: %s = %.12E -> %s = %.12E ->\n          %s = %.12E,  "
                "resid = %.12E\n", skip, from, to, from, spec1[j], to,
                spec2[j], from, clos[j], resid);
         skip = '\0';
      }
   }

   printf("%s%s: Maximum closure residual = %.12E\n", from, to, residmax);
   if (residmax > tol) {
      printf("\n");
      skip = '\0';
   } else {
      skip = '\n';
   }

   return;
}
