/*============================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2002, Mark Calabretta
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation; either version 2 of the License, or (at your
*   option) any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You should have received a copy of the GNU General Public License along
*   with this library; if not, write to the Free Software Foundation, Inc.,
*   675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=============================================================================
*
*   tspc tests the spectral transformation driver routines for closure.
*
*   $Id$
*---------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <cpgplot.h>
#include <wcstrig.h>
#include <spc.h>

#ifndef __STDC__
#ifndef const
#define const
#endif
#endif

const int nspec = 10001;
const double tol = 1.0e-11;

int closure(const char[9], double, double, int, double, double, double);

const double C = 2.99792458e8;

/* KPNO MARS spectrograph grism parameters. */
double mars[7] = {4.5e5, 1.0, 27.0, 1.765, -1.077e6, 3.0, 5.0};


int main()

{
   char text[80];
   int naxisj;
   register int j;
   double cdeltX, crpixj, crvalX, restfrq, restwav, x1, x2;

   /* Uncomment the following two lines to raise SIGFPE on floating point
    * exceptions for the Sun FORTRAN compiler.  This signal can be caught
    * within 'dbx' by issuing the command "catch FPE".
    */
/* #include <floatingpoint.h> */
/* ieee_handler("set", "common", SIGFPE_ABORT); */


   printf("\nTesting closure of WCSLIB spectral transformation routines\n");
   printf("----------------------------------------------------------\n");

   /* List error messages. */
   printf("\nList of spcset error codes:\n");
   for (j = 1; j <= 1 ; j++) {
      printf("   %d: %s.\n", j, spcset_errmsg[j]);
   }

   printf("\nList of spcx2s error codes:\n");
   for (j = 1; j <= 2 ; j++) {
      printf("   %d: %s.\n", j, spcx2s_errmsg[j]);
   }

   printf("\nList of spcs2x error codes:\n");
   for (j = 1; j <= 2 ; j++) {
      printf("   %d: %s.\n", j, spcs2x_errmsg[j]);
   }


   /* PGPLOT initialization. */
   strcpy(text, "/xwindow");
   cpgbeg(0, text, 1, 1);

   naxisj = nspec;
   crpixj = naxisj/2 + 1;

   restfrq = 1420.40595e6;
   restwav = C/restfrq;
   x1 = 1.0e9;
   x2 = 2.0e9;
   cdeltX = (x2 - x1)/(naxisj - 1);
   crvalX = x1 + (crpixj - 1.0)*cdeltX;
   printf("\nLinear frequency axis, span: %.1f to %.1f (GHz), step: %.3f "
          "(kHz)\n---------------------------------------------------------"
          "-----------------\n", x1*1e-9, x2*1e-9, cdeltX*1e-3);
   closure("WAVE-F2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VOPT-F2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ZOPT-F2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AWAV-F2A", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VELO-F2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("BETA-F2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);


   restwav = 700.0e-9;
   restfrq = C/restwav;
   x1 = 300.0e-9;
   x2 = 900.0e-9;
   cdeltX = (x2 - x1)/(naxisj - 1);
   crvalX = x1 + (crpixj - 1.0)*cdeltX;
   printf("\nLinear vacuum wavelength axis, span: %.0f to %.0f (nm), "
          "step: %f (nm)\n---------------------------------------------"
          "-----------------------------\n", x1*1e9, x2*1e9, cdeltX*1e9);
   closure("FREQ-W2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AFRQ-W2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ENER-W2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVN-W2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VRAD-W2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AWAV-W2A", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VELO-W2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("BETA-W2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);


   printf("\nLinear air wavelength axis, span: %.0f to %.0f (nm), "
          "step: %f (nm)\n------------------------------------------"
          "--------------------------------\n", x1*1e9, x2*1e9, cdeltX*1e9);
   closure("FREQ-A2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AFRQ-A2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ENER-A2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVN-A2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VRAD-A2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVE-A2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VOPT-A2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ZOPT-A2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VELO-A2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("BETA-A2V", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);


   restfrq = 1420.40595e6;
   restwav = C/restfrq;
   x1 = -0.96*C;
   x2 =  0.96*C;
   cdeltX = (x2 - x1)/(naxisj - 1);
   crvalX = x1 + (crpixj - 1.0)*cdeltX;
   printf("\nLinear velocity axis, span: %.0f to %.0f m/s, step: %.0f "
          "(m/s)\n------------------------------------------------------"
          "--------------------\n", x1, x2, cdeltX);
   closure("FREQ-V2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AFRQ-V2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ENER-V2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVN-V2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VRAD-V2F", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVE-V2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VOPT-V2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ZOPT-V2W", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AWAV-V2A", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);


   restwav = 650.0e-9;
   restfrq = C/restwav;
   x1 =  300e-9;
   x2 = 1000e-9;
   cdeltX = (x2 - x1)/(naxisj - 1);
   crvalX = x1 + (crpixj - 1.0)*cdeltX;
   printf("\nVacuum wavelength grism axis, span: %.0f to %.0f (nm), "
          "step: %f (nm)\n--------------------------------------------"
          "------------------------------\n", x1*1e9, x2*1e9, cdeltX*1e9);
   closure("FREQ-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AFRQ-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ENER-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVN-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VRAD-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("WAVE-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VOPT-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("ZOPT-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("AWAV-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VELO-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("BETA-GRI", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);


   /* Reproduce Fig. 5 of Paper III. */
   naxisj = 1700;
   crpixj = 719.8;
   crvalX = 7245.2e-10;
   cdeltX = 2.956e-10;
   restwav = 8500.0e-10;
   restfrq = C/restwav;
   x1 = crvalX + (1 - crpixj)*cdeltX;
   x2 = crvalX + (naxisj - crpixj)*cdeltX;
   mars[5] = 0.0;
   mars[6] = 0.0;
   printf("\nAir wavelength grism axis, span: %.0f to %.0f (nm), "
          "step: %f (nm)\n--------------------------------------------"
          "------------------------------\n", x1*1e9, x2*1e9, cdeltX*1e9);
   closure("AWAV-GRA", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);
   closure("VELO-GRA", restfrq, restwav, naxisj, crpixj, cdeltX, crvalX);

   cpgask(0);
   cpgend();

   return 0;
}

/*--------------------------------------------------------------------------*/

int closure (ctype, restfrq, restwav, naxisj, crpixj, cdeltX, crvalX)

const char ctype[9];
int   naxisj;
double cdeltX, crpixj, crvalX, restfrq, restwav;

{
   char ctypeS[5], ctypeX[5], ptype, stype[32], title[80], units[8], xtype,
          ylab[80];
   int stat1[nspec], stat2[nspec], status;
   register int j, k;
   float  tmp, x[nspec], xmin, xmax, y[nspec], ymax, ymin;
   double clos[nspec], cdeltS, crvalS, dPdX, dSdP, resid, residmax,
          spec1[nspec], spec2[nspec], theta;
   struct specvars specs;
   struct spcprm spc;


   spcini(&spc);

   /* Construct a linear axis of the required X-type. */
   xtype = ctype[5];
   if (xtype == 'F') {
      strcpy(ctypeX, "FREQ");
   } else if (xtype == 'W') {
      strcpy(ctypeX, "WAVE");
   } else if (xtype == 'A') {
      strcpy(ctypeX, "AWAV");
   } else if (xtype == 'V') {
      strcpy(ctypeX, "VELO");
   } else if (xtype == 'G') {
      /* KPNO MARS spectrograph grism parameters. */
      spc.pv[0] = mars[0];
      spc.pv[1] = mars[1];
      spc.pv[2] = mars[2];
      spc.pv[3] = mars[3];
      spc.pv[4] = mars[4];
      spc.pv[5] = mars[5];
      spc.pv[6] = mars[6];

      /* cdeltX and crvalX were given as wavelengths... */
      if (ctype[7] == 'I') {
         /* ...in vacuum. */
         xtype = 'W';
         strcpy(ctypeX, "WAVE");
      } else {
         /* ...in air. */
         xtype = 'A';
         strcpy(ctypeX, "AWAV");
      }
   } else {
      return 1;
   }

   specx(ctypeX, crvalX, restfrq, restwav, &specs);

   /* What is the required spectral type? */
   strncpy(ctypeS, ctype, 4);
   ctypeS[4] = '\0';
   if (strcmp(ctypeS, "FREQ") == 0) {
      strcpy(stype, "Frequency");
      strcpy(units, "(Hz)");
      crvalS = specs.freq;
      dSdP = 1.0;
      ptype = 'F';
   } else if (strcmp(ctypeS, "AFRQ") == 0) {
      strcpy(stype, "Angular frequency");
      strcpy(units, "(deg/s)");
      crvalS = specs.afrq;
      dSdP = specs.dafrqfreq;
      ptype = 'F';
   } else if (strcmp(ctypeS, "ENER") == 0) {
      strcpy(stype, "Photon energy");
      strcpy(units, "(J)");
      crvalS = specs.ener;
      dSdP = specs.denerfreq;
      ptype = 'F';
   } else if (strcmp(ctypeS, "WAVN") == 0) {
      strcpy(stype, "Wavenumber");
      strcpy(units, "(1/m)");
      crvalS = specs.wavn;
      dSdP = specs.dwavnfreq;
      ptype = 'F';
   } else if (strcmp(ctypeS, "VRAD") == 0) {
      strcpy(stype, "Radio velocity");
      strcpy(units, "(m/s)");
      crvalS = specs.vrad;
      dSdP = specs.dvradfreq;
      ptype = 'F';
   } else if (strcmp(ctypeS, "WAVE") == 0) {
      strcpy(stype, "Vacuum wavelength");
      strcpy(units, "(m)");
      crvalS = specs.wave;
      dSdP = 1.0;
      ptype = 'W';
   } else if (strcmp(ctypeS, "VOPT") == 0) {
      strcpy(stype, "Optical velocity");
      strcpy(units, "(m/s)");
      crvalS = specs.vopt;
      dSdP = specs.dvoptwave;
      ptype = 'W';
   } else if (strcmp(ctypeS, "ZOPT") == 0) {
      strcpy(stype, "Redshift");
      strcpy(units, "");
      crvalS = specs.zopt;
      dSdP = specs.dzoptwave;
      ptype = 'W';
   } else if (strcmp(ctypeS, "AWAV") == 0) {
      strcpy(stype, "Air wavelength");
      strcpy(units, "(m)");
      crvalS = specs.awav;
      dSdP = 1.0;
      ptype = 'A';
   } else if (strcmp(ctypeS, "VELO") == 0) {
      strcpy(stype, "Relativistic velocity");
      strcpy(units, "(m/s)");
      crvalS = specs.velo;
      dSdP = 1.0;
      ptype = 'V';
   } else if (strcmp(ctypeS, "BETA") == 0) {
      strcpy(stype, "Velocity ratio (v/c)");
      strcpy(units, "");
      crvalS = specs.beta;
      dSdP = specs.dbetavelo;
      ptype = 'V';
   }

   /* Find dP/dX. */
   dPdX = 1.0;
   if (xtype == 'F') {
      if (ptype == 'W') {
         dPdX = specs.dwavefreq;
      } else if (ptype == 'A') {
         dPdX = specs.dawavfreq;
      } else if (ptype == 'V') {
         dPdX = specs.dvelofreq;
      }
   } else if (xtype == 'W') {
      if (ptype == 'F') {
         dPdX = specs.dfreqwave;
      } else if (ptype == 'A') {
         dPdX = specs.dawavwave;
      } else if (ptype == 'V') {
         dPdX = specs.dvelowave;
      }
   } else if (xtype == 'A') {
      if (ptype == 'F') {
         dPdX = specs.dfreqawav;
      } else if (ptype == 'W') {
         dPdX = specs.dwaveawav;
      } else if (ptype == 'V') {
         dPdX = specs.dveloawav;
      }
   } else if (xtype == 'V') {
      if (ptype == 'F') {
         dPdX = specs.dfreqvelo;
      } else if (ptype == 'W') {
         dPdX = specs.dwavevelo;
      } else if (ptype == 'A') {
         dPdX = specs.dawavvelo;
      }
   }

   cdeltS = dSdP * dPdX * cdeltX;

   /* Construct the axis. */
   for (j = 0; j < naxisj; j++) {
      spec1[j] = (j+1 - crpixj)*cdeltS;
   }

   printf("%s (CRVALk+w) range: %13.6E to %13.6E, step: %13.6E\n", ctypeS,
          crvalS+spec1[0], crvalS+spec1[naxisj-1], cdeltS);


   /* Initialize. */
   spc.flag = 0;
   spc.crval = crvalS;
   spc.restfrq = restfrq;
   spc.restwav = restwav;
   strcpy(spc.type, ctypeS);
   strcpy(spc.code, ctype+5);

   /* Convert the first to the second. */
   if (status = spcx2s(&spc, naxisj, 1, 1, spec1, spec2, stat1)) {
      printf("spcx2s: %s.\n", spec_errmsg[status]);
   }

   /* Convert the second back to the first. */
   if (status = spcs2x(&spc, naxisj, 1, 1, spec2, clos, stat2)) {
      printf("spcs2x: %s.\n", spec_errmsg[status]);
   }

   residmax = 0.0;

   /* Test closure. */
   for (j = 0; j < naxisj; j++) {
      if (stat1[j]) {
         printf("%s: w =%20.12E -> %s = ???, stat = %d\n", ctype, spec1[j],
                spc.type, stat1[j]);
         continue;
      }

      if (stat2[j]) {
         printf("%s: w =%20.12E -> %s =%20.12E -> w = ???, stat = %d\n",
                ctype, spec1[j], spc.type, spec2[j], stat2[j]);
         continue;
      }

      resid = fabs((clos[j] - spec1[j])/cdeltS);
      if (resid > residmax) residmax = resid;

      if (resid > tol) {
         printf("%s: w =%20.12E -> %s =%20.12E ->\n"
                "          w =%20.12E,  resid =%20.12E\n", ctype,
                spec1[j], spc.type, spec2[j], clos[j], resid);
      }
   }

   printf("%s: Maximum closure residual = %.12E pixel\n", ctype, residmax);


   /* Draw graph. */
   cpgbbuf();
   cpgeras();

   xmin = (float)(crvalS + spec1[0]);
   xmax = (float)(crvalS + spec1[naxisj-1]);
   ymin = (float)(spec2[0]) - xmin;
   ymax = ymin;
   for (j = 0; j < naxisj; j++) {
      x[j] = (float)(j+1);
      y[j] = (float)(spec2[j] - (crvalS + spec1[j]));
      if (y[j] > ymax) ymax = y[j];
      if (y[j] < ymin) ymin = y[j];
   }

   j = (int)crpixj + 1;
   if (y[j] < 0.0) {
      tmp  = ymin;
      ymin = ymax;
      ymax = tmp;
   }

   cpgask(0);
   cpgenv(1.0f, (float)naxisj, ymin, ymax, 0, -1);

   cpgsci(1);
   cpgbox("ABNTS", 0.0f, 0, "BNTS", 0.0f, 0);
   sprintf(ylab, "%s - correction %s", stype, units);
   sprintf(title, "%s:  CRVALk + w %s", ctype, units);
   cpglab("Pixel coordinate", ylab, title);

   cpgaxis("N", 0.0f, ymax, (float)naxisj, ymax, xmin, xmax, 0.0f, 0,
           -0.5f, 0.0f, 0.5f, -0.5f, 0.0f);

   cpgaxis("N", (float)naxisj, ymin, (float)naxisj, ymax,
           (float)(ymin/cdeltS), (float)(ymax/cdeltS), 0.0f, 0, 0.5f, 0.0f,
           0.5f, 0.1f, 0.0f);
   cpgmtxt("R", 2.2f, 0.5f, 0.5f, "Pixel offset");

   cpgline(naxisj, x, y);
   cpgsci(7);
   cpgpt1((float)crpixj, 0.0f, 24);
   cpgebuf();

   cpgask(1);
   cpgpage();

   printf("\n");

   return 0;
}
