/*=============================================================================
*
*   WCSLIB - an implementation of the FITS World Coordinate System convention.
*   Copyright (C) 1995-2002, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or modify it
*   under the terms of the GNU Library General Public License as published
*   by the Free Software Foundation; either version 2 of the License, or (at
*   your option) any later version.
*
*   This library is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library
*   General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public License
*   along with this library; if not, write to the Free Software Foundation,
*   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=============================================================================
*
*   C routines which implement the FITS World Coordinate System (WCS)
*   convention.
*
*   Summary of routines
*   -------------------
*   These routines apply the linear transformation defined by the WCS FITS
*   header cards.  There are separate routines for the pixel-to-image,
*   linp2x(), and image-to-pixel, linx2p(), transformations.
*
*   An initialization routine, linset(), computes intermediate values from
*   the transformation parameters but need not be called explicitly - see the
*   explanation of lin.flag below.
*
*   Two service routines, linini() and linfree() are provided to manage the
*   linprm struct.  A third, linprt(), prints its contents.
*
*   An auxiliary matrix inversion routine, matinv(), is included.  It uses
*   LU-triangular factorization with scaled partial pivoting.
*
*
*   Service routines for the linprm struct; linini(), & linfree()
*   -------------------------------------------------------------
*   These service routines are provided to manage the linprm struct (see also
*   "Memory allocation and deallocation" below).
*
*   linini() allocates memory for the crpix, pc, and cdelt arrays and sets the
*   members of the linprm struct to default values.
*
*   Given:
*      m_init   const int
*                        If true, the struct is being initialized for the
*                        first time; memory management will be initialized.
*                        It is important not to set this if the struct is
*                        being reinitialized, otherwise memory leaks may
*                        result.
*      alloc    const int
*                        If true, allocate memory for the crpix, pc, and cdelt
*                        arrays.  Otherwise, it is assumed that pointers to
*                        these arrays have been set by the caller except if
*                        they are null pointers in which case memory will be
*                        allocated for them regardless.
*      naxis    const int
*                        The number of world coordinate axes, used to
*                        determine array sizes.
*
*   Returned:
*      lin      struct linprm*
*                        Linear transformation parameters (see below).
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Memory allocation error.
*
*   linfree() frees memory allocated for the linprm arrays by linini() and/or
*   linset().  linini() keeps a record of the memory it allocates and
*   linfree() will only attempt to free this.
*
*   Returned:
*      lin      struct linprm*
*                        Linear transformation parameters (see below).
*
*   Function return value:
*               int      Error status
*                           0: Success.
*
*
*   Print routine for the linprm struct; linprt()
*   ---------------------------------------------
*   This service routine may be used to print the members of a linprm struct.
*
*   Given:
*      lin      const struct linprm*
*                        Linear transformation parameters (see below).
*
*   Function return value:
*               int      Error status
*                           0: Success.
*
*
*   Initialization routine; linset()
*   --------------------------------
*   If necessary, allocates memory for the piximg and imgpix arrays in the
*   linprm struct and initializes the structure according to information
*   supplied within it (see "Linear transformation parameters" below).
*
*   Note that this routine need not be called directly; it will be invoked by
*   linp2x() and linx2p() if the "flag" structure member is anything other
*   than a predefined magic value.
*
*   Given and/or returned:
*      lin      struct linprm*
*                        Linear transformation parameters (see below).
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Memory allocation error.
*                           2: PC matrix is singular.
*
*
*   Pixel-to-world transformation; linp2x()
*   ---------------------------------------
*   Compute image coordinates from pixel coordinates.  Note that where
*   celestial coordinate systems are concerned the image coordinates
*   correspond to (x,y) in the plane of projection, not celestial (lng,lat).
*
*   Given and/or returned:
*      lin      struct linprm*
*                        Linear transformation parameters (see below).
*
*   Given:
*      ncoord   const int
*      nelem    const int
*                        The number of coordinates, each of vector length
*                        nelem but containing lin.naxis coordinate elements.
*      pixcrd   const double[ncoord][nelem]
*                        Array of pixel coordinates.
*
*   Returned:
*      imgcrd   double[ncoord][nelem]
*                        Array of image (world) coordinates.
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Memory allocation error.
*                           2: PC matrix is singular.
*
*
*   World-to-pixel transformation; linx2p()
*   ---------------------------------------
*   Compute pixel coordinates from image coordinates.  Note that where
*   celestial coordinate systems are concerned the image coordinates
*   correspond to (x,y) in the plane of projection, not celestial (lng,lat).
*
*   Given and returned:
*      lin      struct linprm*
*                        Linear transformation parameters (see below).
*
*   Given:
*      ncoord   const int
*      nelem    const int
*                        The number of coordinates, each of vector length
*                        nelem but containing lin.naxis coordinate elements.
*      imgcrd   const double[ncoord][nelem]
*                        Array of image (world) coordinates.
*
*   Returned:
*      pixcrd   double[ncoord][nelem]
*                        Array of pixel coordinates.
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Memory allocation error.
*                           2: PC matrix is singular.
*
*
*   Linear transformation parameters
*   --------------------------------
*   The linprm struct consists of the following:
*
*      int flag
*         This flag must be set to zero whenever any of the following members
*         are set or modified.  This signals the initialization routine,
*         linset(), to recompute intermediaries.
*
*      int naxis
*         Number of pixel and world coordinate elements.
*
*      double *crpix
*         Pointer to the first element of an array of double containing the
*         coordinate reference pixel, CRPIXj.
*
*      double *pc
*         Pointer to the first element of the PC (pixel coordinate)
*         transformation matrix.  The expected order is
*
*            lin.pc = {PC1_1, PC1_2, PC2_1, PC2_2};
*
*         This may be constructed conveniently from a two-dimensional array
*         via
*
*            double m[2][2] = {{PC1_1, PC1_2},
*                              {PC2_1, PC2_2}};
*
*         which is equivalent to,
*
*            double m[2][2];
*            m[0][0] = PC1_1;
*            m[0][1] = PC1_2;
*            m[1][0] = PC2_1;
*            m[1][1] = PC2_2;
*
*         for which the storage order is
*
*            PC1_1, PC1_2, PC2_1, PC2_2
*
*         so it would be legitimate to set lin.pc = *m.
*
*      double *cdelt
*         Pointer to the first element of an array of double containing the
*         coordinate increments, CDELTi.
*
*   The remaining members of the linprm struct are maintained by linset() and
*   must not be modified elsewhere:
*
*      int unity
*         True if the linear transformation matrix is unity.
*
*      double *piximg
*         Pointer to the first element of the matrix containing the product
*         of the CDELTi diagonal matrix and the PC matrix.
*
*      double *imgpix
*         Pointer to the first element of the inverse of the piximg matrix.
*
*      int m_flag, m_naxis
*      double *m_crpix, *m_pc, *m_cdelt
*         These are used for memory management by linini() and linfree().
*
*
*   Vector arguments
*   ----------------
*   Arrays of pixel and world coordinates are two dimensional, i.e.
*   pixcrd[ncoord][nelem], where nelem must equal or exceed the number of
*   coordinate elements, naxis, stored in the linprm struct.  Exception: when
*   ncoord == 1, nelem is not used.
*
*   Note that the function prototypes must declare two-dimensional arrays as
*   one-dimensional to avoid warnings about declaration of "incomplete types".
*   This was considered preferble to declaring them as simple
*   pointers-to-double which gives no indication that storage is associated
*   with them.
*
*
*   Memory allocation and deallocation
*   ----------------------------------
*   linini() allocates memory for the crpix, pc, and cdelt arrays in the
*   linprm struct.  It is provided as a service routine; usage is optional,
*   and the caller is at liberty to set these pointers independently.
*
*   If the pc matrix is not unity, linset() also allocates memory for the
*   piximg and imgpix arrays.  The caller must not modify these.
*
*   linini() maintains a record of memory it has allocated and this is used
*   by linfree() which linini() uses to free any memory which it may have
*   allocated on a previous invokation.  Thus it is not necessary for the
*   caller to invoke linfree() separately.  Likewise, linset() deallocates
*   memory which it may have allocated on a previous invokation.
*
*   However, a memory leak will result if a linprm struct goes out of scope
*   before the memory has been free'd, either by linfree() or otherwise.
*   Likewise, if the linprm struct itself has been malloc'd and the allocated
*   memory is not free'd when the memory for the struct is free'd.  A leak may
*   also arise if the caller interferes with the array pointers in the
*   "private" part of the linprm struct.
*
*   Beware of making a shallow copy of a linprm struct by assignment.  When
*   the memory allocated for one is free'd the copy will reference unallocated
*   memory.
*
*
*   Error codes
*   -----------
*   Error messages to match the error codes for linini(), linset(), linp2x(),
*   and linx2p() are encoded in the linini_errmsg, linset_errmsg,
*   linp2x_errmsg, and linx2p_errmsg character arrays.
*
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: lin.h,v 2.6 2001/11/15 03:14:40 mcalabre Exp $
*===========================================================================*/

#ifndef WCSLIB_LIN
#define WCSLIB_LIN

#ifdef __cplusplus
extern "C" {
#endif

#if !defined(__STDC__) && !defined(__cplusplus)
#ifndef const
#define const
#endif
#endif


extern const char *linini_errmsg[];
extern const char *linset_errmsg[];
extern const char *linp2x_errmsg[];
extern const char *linx2p_errmsg[];


struct linprm {
   /* Initialization flag (see the prologue above).                         */
   /*-----------------------------------------------------------------------*/
   int flag;			/* Set to zero to force initialization.     */

   /* Parameters to be provided (see the prologue above).                   */
   /*-----------------------------------------------------------------------*/
   int naxis;			/* The number of axes, given by NAXIS.      */
   double *crpix;		/* CRPIXj cards for each pixel axis.        */
   double *pc;			/* PCi_j  linear transformation matrix.     */
   double *cdelt;		/* CDELTi cards for each coordinate axis.   */

   /* Information derived from the parameters supplied.                     */
   /*-----------------------------------------------------------------------*/
   int unity;			/* True if the PCi_j matrix is unity.       */
   double *piximg;		/* Product of the CDELTi and PCij matrices. */
   double *imgpix;		/* Inverse of the piximg matrix.            */

   int m_flag, m_naxis;		/* The remainder are for memory management. */
   double *m_crpix, *m_pc, *m_cdelt;
};

#if __STDC__  || defined(__cplusplus)
   int linini(const int, const int, const int, struct linprm *);

   int linfree(struct linprm *);

   int linprt(const struct linprm *);

   int linset(struct linprm *);

   int linp2x(struct linprm *, const int, const int, const double[],
              double[]);

   int linx2p(struct linprm *, const int, const int, const double[],
              double[]);

   int matinv(const int, const double [], double []);
#else
   int linini(), linfree(), linprt(), linset(), linp2x(), linx2p(), matinv();
#endif


/* Define macros for scalar invokation for compatibility with WCSLIB 2.x. */
extern const char *linfwd_errmsg[];
extern const char *linrev_errmsg[];
#define linrev(pixcrd, lin, imgcrd) linp2x(lin, 1, 1, pixcrd, imgcrd)
#define linfwd(imgcrd, lin, pixcrd) linx2p(lin, 1, 1, imgcrd, pixcrd)

#ifdef __cplusplus
};
#endif

#endif /* WCSLIB_LIN */
