/*============================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995-2002, Mark Calabretta
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation; either version 2 of the License, or (at your
*   option) any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You should have received a copy of the GNU General Public License along
*   with this library; if not, write to the Free Software Foundation, Inc.,
*   675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=============================================================================
*
*   twcs2 tests wcsmix() for closure on the 1 degree celestial grid for a
*   number of selected projections.  Points with good solutions are marked
*   with a white dot on a graphical display of the projection while bad
*   solutions are flagged with a red circle.
*
*   $Id: twcs2.c,v 2.16 2001/09/05 05:07:17 mcalabre Exp $
*---------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <cpgplot.h>
#include <sph.h>
#include <wcs.h>

#ifndef __STDC__
#ifndef const
#define const
#endif
#endif

void grdplt(double, double, double, double);
void parser();
void mixex();
void id(int *, double, double, double, double);

/* Set to 1 to skip mixex(), primarily for debugging purposes. */
const int skip_mixex = 0;

/* Reporting tolerance for mixex(). */
const double tol = 1.0e-9;


/* In real life these would be encoded as FITS header cards. */
const int NAXIS = 4;
const double CRPIX[4] =  {  513.0,  0.0,  0.0,  0.0};
const double PC[4][4] = {{    1.1,  0.0,  0.0,  0.0},
                         {    0.0,  1.0,  0.0,  0.1},
                         {    0.0,  0.0,  1.0,  0.0},
                         {    0.0,  0.2,  0.0,  1.0}};
const double CDELT[4] =  {62500.0,  1.0,  1.0, -1.0};

				/* The "xxx" is reset in main(). */
char CTYPE[4][9] = {"WAVE-F2W", "XLAT-xxx", "TIME    ", "XLON-xxx"};

				/* Will be reset in grdplt(). */
double CRVAL[4] =  {1.3945e9, 0.0, -2e3, 0.0};
double LONPOLE  = 999.0;
double LATPOLE  = 999.0;
double RESTFRQ  =   1.42040575e9;
double RESTWAV  =   0.0;

int NPV;
struct pvcard PV[10];		/* Projection parameters are set in main(). */

struct wcsprm wcs;

int main()

{
   char   text[80];
   register int j;

   /* Uncomment the following two lines to raise SIGFPE on floating point
    * exceptions for the Sun FORTRAN compiler.  This signal can be caught
    * within 'dbx' by issuing the command "catch FPE".
    */
/* #include <floatingpoint.h> */
/* ieee_handler("set", "common", SIGFPE_ABORT); */


   printf("\nTesting WCSLIB wcsmix routine\n");
   printf("-----------------------------\n");

   /* List error messages. */
   printf("\nList of wcsini error codes:\n");
   for (j = 1; j <= 1 ; j++) {
      printf("   %d: %s.\n", j, wcsini_errmsg[j]);
   }

   printf("\nList of wcsset error codes:\n");
   for (j = 1; j <= 6 ; j++) {
      printf("   %d: %s.\n", j, wcsset_errmsg[j]);
   }

   printf("\nList of wcss2x error codes:\n");
   for (j = 1; j <= 7 ; j++) {
      printf("   %d: %s.\n", j, wcss2x_errmsg[j]);
   }

   printf("\nList of wcsx2s error codes:\n");
   for (j = 1; j <= 7 ; j++) {
      printf("   %d: %s.\n", j, wcsx2s_errmsg[j]);
   }

   printf("\nList of wcsmix error codes:\n");
   for (j = 1; j <= 8 ; j++) {
      printf("   %d: %s.\n", j, wcsmix_errmsg[j]);
   }


   /* PGPLOT initialization. */
   strcpy(text, "/xwindow");
   cpgbeg(0, text, 1, 1);

   /* Define pen colours. */
   cpgscr(0, 0.00f, 0.00f, 0.00f);
   cpgscr(1, 1.00f, 1.00f, 0.00f);
   cpgscr(2, 1.00f, 1.00f, 1.00f);
   cpgscr(3, 0.50f, 0.50f, 0.80f);
   cpgscr(4, 0.80f, 0.50f, 0.50f);
   cpgscr(5, 0.80f, 0.80f, 0.80f);
   cpgscr(6, 0.50f, 0.50f, 0.80f);
   cpgscr(7, 0.80f, 0.50f, 0.50f);
   cpgscr(8, 0.30f, 0.50f, 0.30f);
   cpgscr(9, 1.00f, 0.75f, 0.00f);


   /* Initialize the wcsprm struct. */
   wcsini(1, 1, NAXIS, &wcs);

   /*----------------------------------------------------------*/
   /* Set the PVi_m cards for the longitude axis so that the   */
   /* fiducial native coordinates are at the native pole, i.e. */
   /* (phi0,theta0) = (0,90), but without any fiducial offset. */
   /* We do this as a test, and also so that all projections   */
   /* will be exercised with the same obliquity parameters.    */
   /*----------------------------------------------------------*/
   PV[0].i = 4;			/* Longitude axis at j = 4.    */
   PV[0].m = 1;			/* Parameter number.           */
   PV[0].value =  0.0;		/* Fiducial native longitude.  */

   PV[1].i = 4;			/* Longitude axis at j = 4.    */
   PV[1].m = 2;			/* Parameter number.           */
   PV[1].value = 90.0;		/* Fiducial native latitude.   */

   /* Set the PVi_m cards for the latitude axes.               */
   PV[2].i = 2;			/* Latitude axis at j = 2.     */
   PV[2].m = 1;			/* Parameter number.           */
   PV[2].value = 0.0;		/* PVi_1 (reset below).        */

   PV[3].i = 2;			/* Latitude axis at j = 2.     */
   PV[3].m = 2;			/* Parameter number.           */
   PV[3].value = 0.0;		/* PVi_2 (reset below).        */


   /* ARC: zenithal/azimuthal equidistant. */
   strncpy(&CTYPE[1][5], "ARC", 3);
   strncpy(&CTYPE[3][5], "ARC", 3);
   NPV = 2;
   grdplt(-190.0, 190.0, -190.0, 190.0);

   /* ZEA: zenithal/azimuthal equal area. */
   strncpy(&CTYPE[1][5], "ZEA", 3);
   strncpy(&CTYPE[3][5], "ZEA", 3);
   NPV = 2;
   grdplt(-120.0, 120.0, -120.0, 120.0);

   /* CYP: cylindrical perspective. */
   strncpy(&CTYPE[1][5], "CYP", 3);
   strncpy(&CTYPE[3][5], "CYP", 3);
   NPV = 4;
   PV[2].value = 3.0;
   PV[3].value = 0.8;
   grdplt(-170.0, 170.0, -170.0, 170.0);

   /* CEA: cylindrical equal area. */
   strncpy(&CTYPE[1][5], "CEA", 3);
   strncpy(&CTYPE[3][5], "CEA", 3);
   NPV = 3;
   PV[2].value = 0.75;
   grdplt(-200.0, 200.0, -200.0, 200.0);

   /* CAR: Cartesian. */
   strncpy(&CTYPE[1][5], "CAR", 3);
   strncpy(&CTYPE[3][5], "CAR", 3);
   NPV = 2;
   grdplt(-210.0, 210.0, -210.0, 210.0);

   /* SFL: Sanson-Flamsteed. */
   strncpy(&CTYPE[1][5], "SFL", 3);
   strncpy(&CTYPE[3][5], "SFL", 3);
   NPV = 2;
   grdplt(-190.0, 190.0, -190.0, 190.0);

   /* PAR: parabolic. */
   strncpy(&CTYPE[1][5], "PAR", 3);
   strncpy(&CTYPE[3][5], "PAR", 3);
   NPV = 2;
   grdplt(-190.0, 190.0, -190.0, 190.0);

   /* MOL: Mollweide's projection. */
   strncpy(&CTYPE[1][5], "MOL", 3);
   strncpy(&CTYPE[3][5], "MOL", 3);
   NPV = 2;
   grdplt(-170.0, 170.0, -170.0, 170.0);

   /* AIT: Hammer-Aitoff. */
   strncpy(&CTYPE[1][5], "AIT", 3);
   strncpy(&CTYPE[3][5], "AIT", 3);
   NPV = 2;
   grdplt(-170.0, 170.0, -170.0, 170.0);

   /* COE: conic equal area. */
   strncpy(&CTYPE[1][5], "COE", 3);
   strncpy(&CTYPE[3][5], "COE", 3);
   NPV = 4;
   PV[2].value = 60.0;
   PV[3].value = 15.0;
   grdplt(-140.0, 140.0, -120.0, 160.0);

   /* COD: conic equidistant. */
   strncpy(&CTYPE[1][5], "COD", 3);
   strncpy(&CTYPE[3][5], "COD", 3);
   NPV = 4;
   PV[2].value = 60.0;
   PV[3].value = 15.0;
   grdplt(-200.0, 200.0, -180.0, 220.0);

   /* BON: Bonne's projection. */
   strncpy(&CTYPE[1][5], "BON", 3);
   strncpy(&CTYPE[3][5], "BON", 3);
   NPV = 3;
   PV[2].value = 30.0;
   grdplt(-160.0, 160.0, -160.0, 160.0);

   /* PCO: polyconic. */
   strncpy(&CTYPE[1][5], "PCO", 3);
   strncpy(&CTYPE[3][5], "PCO", 3);
   NPV = 2;
   grdplt(-190.0, 190.0, -190.0, 190.0);

   /* TSC: tangential spherical cube. */
   strncpy(&CTYPE[1][5], "TSC", 3);
   strncpy(&CTYPE[3][5], "TSC", 3);
   NPV = 2;
   grdplt(-340.0, 80.0, -210.0, 210.0);

   /* QSC: quadrilateralized spherical cube. */
   strncpy(&CTYPE[1][5], "QSC", 3);
   strncpy(&CTYPE[3][5], "QSC", 3);
   NPV = 2;
   grdplt(-340.0, 90.0, -210.0, 210.0);

   cpgend();

   return 0;
}

/*--------------------------------------------------------------------------*/

void grdplt(imin, imax, jmin, jmax)

double imax, imin, jmax, jmin;

{
   const int NELEM = 9;

   char   text[80];
   int    ci, i, ilat, ilng, k, stat[361];
   float  fimax, fimin, fjmax, fjmin, ir[1024], jr[1024];
   double freq, lat, lng, phi[361], theta[361], img[361][NELEM], step,
          pix[361][NELEM], world[361][NELEM];
   double *pixlat, *pixlng, *worldlat, *worldlng, *worldp;


   /* This routine simulates the actions of a FITS header parser. */
   CRVAL[1] =  90.0;
   CRVAL[3] =   0.0;
   LONPOLE  = 999.0;
   parser();


   /* Initialize non-celestial world coordinates. */
   freq = 1.42040595e9 - 180.0 * 62500.0;
   for (k = 0; k < 361; k++) {
      world[k][0] = 0.0;
      world[k][1] = 0.0;
      world[k][2] = 0.0;
      world[k][3] = 0.0;

      world[k][wcs.spec] = 2.99792458e8 / freq;
      freq += 62500.0;
   }


   /* Define PGPLOT viewport. */
   fimax = (float)imax;
   fimin = (float)imin;
   fjmax = (float)jmax;
   fjmin = (float)jmin;
   cpgenv(fimin, fimax, fjmin, fjmax, 1, -2);

   /* Draw face boundaries of the quad-cube projections. */
   if (wcs.cel.prj.category == QUADCUBE) {
      cpgsci(8);

      pixlng = pix[0] + wcs.lng;
      pixlat = pix[0] + wcs.lat;
      worldlng = world[0] + wcs.lng;
      worldlat = world[0] + wcs.lat;

      /* Draw the map boundary. */
      img[0][0] = 0.0;
      img[0][1] = 0.0;
      img[0][2] = 0.0;
      img[0][3] = 0.0;

      img[0][wcs.lng] = -wcs.cel.prj.w[0];
      img[0][wcs.lat] =  wcs.cel.prj.w[0];
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[0] = *pixlng;
      jr[0] = *pixlat;

      img[0][wcs.lng] = -wcs.cel.prj.w[0];
      img[0][wcs.lat] =  wcs.cel.prj.w[0]*3.0;
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[1] = *pixlng;
      jr[1] = *pixlat;

      img[0][wcs.lng] =  wcs.cel.prj.w[0];
      img[0][wcs.lat] =  wcs.cel.prj.w[0]*3.0;
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[2] = *pixlng;
      jr[2] = *pixlat;

      img[0][wcs.lng] =  wcs.cel.prj.w[0];
      img[0][wcs.lat] = -wcs.cel.prj.w[0]*3.0;
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[3] = *pixlng;
      jr[3] = *pixlat;

      img[0][wcs.lng] = -wcs.cel.prj.w[0];
      img[0][wcs.lat] = -wcs.cel.prj.w[0]*3.0;
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[4] = *pixlng;
      jr[4] = *pixlat;

      img[0][wcs.lng] = -wcs.cel.prj.w[0];
      img[0][wcs.lat] =  wcs.cel.prj.w[0];
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[5] = *pixlng;
      jr[5] = *pixlat;

      img[0][wcs.lng] =  wcs.cel.prj.w[0]*7.0;
      img[0][wcs.lat] =  wcs.cel.prj.w[0];
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[6] = *pixlng;
      jr[6] = *pixlat;

      img[0][wcs.lng] =  wcs.cel.prj.w[0]*7.0;
      img[0][wcs.lat] = -wcs.cel.prj.w[0];
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[7] = *pixlng;
      jr[7] = *pixlat;

      img[0][wcs.lng] = -wcs.cel.prj.w[0];
      img[0][wcs.lat] = -wcs.cel.prj.w[0];
      linx2p(&(wcs.lin), 1, 0, img[0], pix[0]);
      ir[8] = *pixlng;
      jr[8] = *pixlat;

      cpgline(9, ir, jr);
   }


   /* Draw the native coordinate grid faintly in the background. */
   cpgsci(8);

   if (wcs.cel.prj.category == POLYCONIC) {
      step = 10.0;
   } else {
      step = 15.0;
   }

   /* Draw native meridians of longitude. */
   for (ilng = -180; ilng <= 180; ilng += 15) {
      lng = (double)ilng;
      if (ilng == -180) lng = -179.99;
      if (ilng ==  180) lng =  179.99;

      worldlng = world[0] + wcs.lng;
      worldlat = world[0] + wcs.lat;
      for (ilat = -90; ilat <= 90; ilat++) {
         lat = (double)ilat;

         *worldlng = lng;
         *worldlat = lat;

         worldlng += NELEM;
         worldlat += NELEM;
      }

      if (wcss2x(&wcs, 181, NELEM, world[0], phi, theta, img[0], pix[0],
                 stat)) {
         continue;
      }

      k = 0;
      pixlng = pix[0] + wcs.lng;
      pixlat = pix[0] + wcs.lat;
      for (ilat = -90; ilat <= 90; ilat++) {
         if (wcs.cel.prj.category == QUADCUBE && k > 0) {
            if (fabs(*pixlng - ir[k-1]) > 2.0 ||
                fabs(*pixlat - jr[k-1]) > 5.0) {
               if (k > 1) cpgline(k, ir, jr);
               k = 0;
            }
         }

         ir[k] = *pixlng;
         jr[k] = *pixlat;
         k++;

         pixlng += NELEM;
         pixlat += NELEM;
      }

      cpgline(k, ir, jr);
   }

   /* Draw native parallels of latitude. */
   for (ilat = -90; ilat <= 90; ilat += 15) {
      lat = (double)ilat;

      worldlng = world[0] + wcs.lng;
      worldlat = world[0] + wcs.lat;
      for (ilng = -180; ilng <= 180; ilng++) {
         lng = (double)ilng;
         if (ilng == -180) lng = -179.99;
         if (ilng ==  180) lng =  179.99;

         *worldlng = lng;
         *worldlat = lat;

         worldlng += NELEM;
         worldlat += NELEM;
      }

      if (wcss2x(&wcs, 361, NELEM, world[0], phi, theta, img[0], pix[0],
                 stat)) {
         continue;
      }

      k = 0;
      pixlng = pix[0] + wcs.lng;
      pixlat = pix[0] + wcs.lat;
      for (ilng = -180; ilng <= 180; ilng++) {
         if (wcs.cel.prj.category == QUADCUBE && k > 0) {
            if (fabs(*pixlng - ir[k-1]) > 2.0 ||
                fabs(*pixlat - jr[k-1]) > 5.0) {
               if (k > 1) cpgline(k, ir, jr);
               k = 0;
            }
         }

         ir[k] = *pixlng;
         jr[k] = *pixlat;
         k++;

         pixlng += NELEM;
         pixlat += NELEM;
      }

      cpgline(k, ir, jr);
   }


   /* Draw a colour-coded celestial coordinate grid. */
   CRVAL[1] = -30.0;
   CRVAL[3] = 150.0;
   LONPOLE  = 150.0;
   parser();

   ci = 1;

   /* Draw celestial meridians of longitude. */
   for (ilng = -180; ilng <= 180; ilng += 15) {
      lng = (double)ilng;

      if (++ci > 7) ci = 2;
      cpgsci(ilng?ci:1);

      worldlng = world[0] + wcs.lng;
      worldlat = world[0] + wcs.lat;
      for (ilat = -90; ilat <= 90; ilat++) {
         lat = (double)ilat;

         *worldlng = lng;
         *worldlat = lat;

         worldlng += NELEM;
         worldlat += NELEM;
      }

      if (wcss2x(&wcs, 181, NELEM, world[0], phi, theta, img[0], pix[0],
                 stat)) {
         continue;
      }

      k = 0;
      pixlng = pix[0] + wcs.lng;
      pixlat = pix[0] + wcs.lat;
      for (ilat = -90; ilat <= 90; ilat++) {
         /* Test for discontinuities. */
         if (k > 0) {
            if (fabs(*pixlng - ir[k-1]) > step ||
                fabs(*pixlat - jr[k-1]) > step) {
               if (k > 1) cpgline(k, ir, jr);
               k = 0;
            }
         }

         ir[k] = *pixlng;
         jr[k] = *pixlat;
         k++;

         pixlng += NELEM;
         pixlat += NELEM;
      }

      cpgline(k, ir, jr);
   }

   /* Draw celestial parallels of latitude. */
   ci = 1;
   for (ilat = -90; ilat <= 90; ilat += 15) {
      lat = (double)ilat;

      if (++ci > 7) ci = 2;
      cpgsci(ilat?ci:1);

      worldlng = world[0] + wcs.lng;
      worldlat = world[0] + wcs.lat;
      for (ilng = -180; ilng <= 180; ilng++) {
         lng = (double)ilng;

         *worldlng = lng;
         *worldlat = lat;

         worldlng += NELEM;
         worldlat += NELEM;
      }

      if (wcss2x(&wcs, 361, NELEM, world[0], phi, theta, img[0], pix[0],
                 stat)) {
         continue;
      }

      k = 0;
      pixlng = pix[0] + wcs.lng;
      pixlat = pix[0] + wcs.lat;
      for (ilng = -180; ilng <= 180; ilng++) {
         /* Test for discontinuities. */
         if (k > 0) {
            if (fabs(*pixlng - ir[k-1]) > step ||
                fabs(*pixlat - jr[k-1]) > step) {
               if (k > 1) cpgline(k, ir, jr);
               k = 0;
            }
         }

         ir[k] = *pixlng;
         jr[k] = *pixlat;
         k++;

         pixlng += NELEM;
         pixlat += NELEM;
      }

      cpgline(k, ir, jr);
   }


   /* Write a descriptive title. */
   cpgsci(1);
   sprintf(text, "%s projection - 15 degree graticule", wcs.cel.prj.code);
   printf("\n%s\n", text);
   fjmin = jmin - 10.0;
   cpgtext(fimin, fjmin, text);

   sprintf(text, "centered on celestial coordinates (%6.2f,%6.2f)",
      wcs.cel.ref[0], wcs.cel.ref[1]);
   printf("%s\n", text);
   fjmin = jmin - 20.0;
   cpgtext(fimin, fjmin, text);

   sprintf(text, "with celestial pole at native coordinates (%7.2f,%7.2f)",
      wcs.cel.ref[2], wcs.cel.ref[3]);
   printf("%s\n", text);
   fjmin = jmin - 30.0;
   cpgtext(fimin, fjmin, text);

   cpgsci(2);


   if (!skip_mixex) mixex();

   return;
}

/*--------------------------------------------------------------------------*/

void parser()

{
   int i, j;
   double *pcij;

   /* In practice a parser would read the FITS header until it encountered  */
   /* the NAXIS card; this must occur near the start of the header before   */
   /* any of the WCS keywords.  In this simulation the header keyvalues are */
   /* set as global variables.  It then uses wcsini() to allocate memory    */
   /* for arrays in the wcsprm struct and set default values.               */
   /*                                                                       */
   /* The global wcsprm struct, first initialized in main(), is being       */
   /* reinitialized here.  Hence the m_init argument is zero.               */
   wcsini(0, 1, NAXIS, &wcs);


   /* Now the parser scans the FITS header, identifying WCS keywords and    */
   /* loading their values into the appropriate elements of the wcsprm      */
   /* struct.                                                               */

   for (j = 0; j < NAXIS; j++) {
      wcs.crpix[j] = CRPIX[j];
   }

   pcij = wcs.pc;
   for (i = 0; i < NAXIS; i++) {
      for (j = 0; j < NAXIS; j++) {
         *(pcij++) = PC[i][j];
      }
   }

   for (i = 0; i < NAXIS; i++) {
      wcs.cdelt[i] = CDELT[i];
   }

   for (i = 0; i < NAXIS; i++) {
      strcpy(wcs.ctype[i], &CTYPE[i][0]);
   }

   for (i = 0; i < NAXIS; i++) {
      wcs.crval[i] = CRVAL[i];
   }

   wcs.lonpole = LONPOLE;
   wcs.latpole = LATPOLE;

   wcs.restfrq = RESTFRQ;
   wcs.restwav = RESTWAV;

   wcs.npv = NPV;
   for (i = 0; i < NPV; i++) {
      wcs.pv[i] = PV[i];
   }

   /* Extract information from the FITS header. */
   (void) wcsset(&wcs);

   return;
}


/*----------------------------------------------------------------------------
*   mixex() tests wcsmix().
*---------------------------------------------------------------------------*/

void mixex()

{
   int    doid, lat, lng, wcslat, wcslng, stat, status;
   register int k;
   float  ipt[1], jpt[1];
   double lng1, lat1, phi, theta;
   double latspan[2], lngspan[2];
   double img[4], pix1[4], pix2[4], pix3[4], world[4];
   double pixlng, pixlat, *worldlat, *worldlng;


   /* Cache frequently used values. */
   wcslng = wcs.lng;
   wcslat = wcs.lat;
   worldlng = world + wcslng;
   worldlat = world + wcslat;

   printf("Testing %s; reporting tolerance %5.1g deg.\n", wcs.cel.prj.code,
          tol);

   world[0] = 0.0;
   world[1] = 0.0;
   world[2] = 0.0;
   world[3] = 0.0;

   world[wcs.spec] = 2.99792458e8 / RESTFRQ;

   for (lat = 90; lat >= -90; lat--) {
      lat1 = (double)lat;

      for (lng = -180; lng <= 180; lng+=15) {
         lng1 = (double)lng;

         *worldlng = lng1;
         *worldlat = lat1;
         if (status = wcss2x(&wcs, 1, 4, world, &phi, &theta, img, pix1,
                             &stat)) {
            printf("%3s(s2x): lng1 =%20.15f  lat1 =%20.15f  error %3d\n",
               wcs.cel.prj.code, lng1, lat1, status);
            continue;
         }

         pixlng = pix1[wcslng];
         pixlat = pix1[wcslat];

         ipt[0] = pixlng;
         jpt[0] = pixlat;
         cpgpt(1, ipt, jpt, -1);

         lngspan[0] = lng1 - 9.3;
         if (lngspan[0] < -180.0) lngspan[0] = -180.0;
         lngspan[1] = lng1 + 4.1;
         if (lngspan[1] >  180.0) lngspan[1] =  180.0;
         latspan[0] = lat1 - 3.7;
         if (latspan[0] <  -90.0) latspan[0] =  -90.0;
         latspan[1] = lat1 + 7.2;
         if (latspan[1] >   90.0) latspan[1] =   90.0;

         doid = 1;

         pix2[wcslng] = pixlng;
         if (status = wcsmix(&wcs, wcslng, 1, latspan, 1.0, 0, world, &phi,
                             &theta, img, pix2)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  A: wcsmix error %d, %s.\n", status,
                   wcsmix_errmsg[status]);

         } else if (status = wcss2x(&wcs, 1, 0, world, &phi, &theta, img,
                                 pix3, &stat)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  A: wcss2x error %d, %s.\n", status,
                   wcss2x_errmsg[status]);

         } else if (fabs(pix3[wcslng]-pixlng) > tol &&
                   (fabs(*worldlat-lat1)  > tol ||
                    fabs(pix2[wcslat]-pixlat) > tol)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  A: (lng2) =%20.15f   lat2  =%20.15f\n", *worldlng,
               *worldlat);
            printf("       phi  =%20.15f  theta  =%20.15f\n", phi, theta);
            printf("       (i2) =%20.15f     j2  =%20.15f\n", pix2[wcslng],
               pix2[wcslat]);
         }


         pix2[wcslat] = pixlat;
         if (status = wcsmix(&wcs, wcslat, 1, latspan, 1.0, 0, world, &phi,
                             &theta, img, pix2)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  B: wcsmix error %d, %s.\n", status,
                   wcsmix_errmsg[status]);

         } else if (status = wcss2x(&wcs, 1, 0, world, &phi, &theta, img,
                                    pix3, &stat)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  B: wcss2x error %d, %s.\n", status,
                   wcss2x_errmsg[status]);

         } else if (fabs(pix3[wcslat]-pixlat) > tol &&
                   (fabs(*worldlat-lat1)  > tol ||
                    fabs(pix2[wcslng]-pixlng) > tol)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  B: (lng2) =%20.15f   lat2  =%20.15f\n", *worldlng,
               *worldlat);
            printf("       phi  =%20.15f  theta  =%20.15f\n", phi, theta);
            printf("        i2  =%20.15f    (j2) =%20.15f\n", pix2[wcslng],
               pix2[wcslat]);
         }

         *worldlat = lat1;

         pix2[wcslng] = pixlng;
         if (status = wcsmix(&wcs, wcslng, 2, lngspan, 1.0, 0, world, &phi,
                             &theta, img, pix2)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  C: wcsmix error %d, %s.\n", status,
                   wcsmix_errmsg[status]);

         } else if (status = wcss2x(&wcs, 1, 0, world, &phi, &theta, img,
                                    pix3, &stat)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  C: wcss2x error %d, %s.\n", status,
                   wcss2x_errmsg[status]);

         } else if (fabs(pix3[wcslng]-pixlng) > tol &&
                   (fabs(*worldlng-lng1)  > tol ||
                    fabs(pix2[wcslat]-pixlat) > tol)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  C:  lng2  =%20.15f  (lat2) =%20.15f\n", *worldlng,
               *worldlat);
            printf("       phi  =%20.15f  theta  =%20.15f\n", phi, theta);
            printf("       (i2) =%20.15f     j2  =%20.15f\n", pix2[wcslng],
               pix2[wcslat]);
         }


         pix2[wcslat] = pixlat;
         if (status = wcsmix(&wcs, wcslat, 2, lngspan, 1.0, 0, world, &phi,
                             &theta, img, pix2)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  D: wcsmix error %d, %s.\n", status,
                   wcsmix_errmsg[status]);

         } else if (status = wcss2x(&wcs, 1, 0, world, &phi, &theta, img,
                                    pix3, &stat)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  D: wcss2x error %d, %s.\n", status,
                   wcss2x_errmsg[status]);

         } else if (fabs(pix3[wcslat]-pixlat) > tol &&
                   (fabs(*worldlng-lng1)  > tol ||
                    fabs(pix2[wcslng]-pixlng) > tol)) {
            id(&doid, lng1, lat1, pixlng, pixlat);
            printf("  D:  lng2  =%20.15f  (lat2) =%20.15f\n", *worldlng,
               *worldlat);
            printf("       phi  =%20.15f  theta  =%20.15f\n", phi, theta);
            printf("        i2  =%20.15f    (j2) =%20.15f\n", pix2[wcslng],
               pix2[wcslat]);
         }

      }
   }

   return;
}


void id(doid, lng1, lat1, pixlng, pixlat)

int    *doid;
double lng1, lat1, pixlng, pixlat;

{
   float  ipt[1], jpt[1];
   double phi, theta;

   if (*doid) {
      /* Compute native coordinates. */
      sphs2x(wcs.cel.euler, 1, 1, 1, 1, &lng1, &lat1, &phi, &theta);

      printf("\n%3s:  lng1  =%20.15f   lat1  =%20.15f\n", wcs.cel.prj.code,
         lng1, lat1);
      printf(  "       phi  =%20.15f  theta  =%20.15f\n", phi, theta);
      printf(  "        i1  =%20.15f     j1  =%20.15f\n", pixlng, pixlat);
      *doid = 0;

      cpgsci(9);
      ipt[0] = pixlng;
      jpt[0] = pixlat;
      cpgpt(1, ipt, jpt, 21);
      cpgsci(2);
   }

   return;
}
