*=======================================================================
*
*   WCSLIB 3.0 - an implementation of the FITS WCS convention.
*   Copyright (C) 1995-2003, Mark Calabretta
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this library; if not, write to the Free Software
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta
*                      Australia Telescope National Facility, CSIRO
*                      PO Box 76
*                      Epping NSW 1710
*                      AUSTRALIA
*
*=======================================================================
      PROGRAM TWCS1
*-----------------------------------------------------------------------
*
*   TWCS1 tests WCSS2P and WCSP2S for closure on the 1 degree celestial
*   graticule for a number of selected projections.
*
*   $Id: twcs1.f,v 3.0 2003/04/01 05:10:21 mcalabre Exp $
*-----------------------------------------------------------------------
*     Number of axes.
      INTEGER   N
      PARAMETER (N = 4)

      INTEGER   I, J, K, NAXIS, NPV, PVI(4), PVM(4)
      DOUBLE PRECISION CDELT(N), CRPIX(N), CRVAL(N), LATPOLE, LONPOLE,
     :          PC(N,N), PV(4), RESTFRQ, RESTWAV
      CHARACTER CTYPE(N)*72

      DOUBLE PRECISION TOL
      PARAMETER (TOL = 1D-9)

      COMMON /HEADER/ NAXIS, NPV, CRPIX, PC, CDELT, CRVAL, LONPOLE,
     :                LATPOLE, RESTFRQ, RESTWAV, PVI, PVM, PV
      COMMON /HEADCH/ CTYPE

      DATA NAXIS   /N/
      DATA (CRPIX(I), I=1,N)
     :             /513D0,   0D0,   0D0,   0D0/
      DATA ((PC(I,J),J=1,N),I=1,N)
     :             /1.1D0,   0D0,   0D0,   0D0,
     :                0D0, 1.0D0,   0D0, 0.1D0,
     :                0D0,   0D0, 1.0D0,   0D0,
     :                0D0, 0.2D0,   0D0, 1.0D0/
      DATA (CDELT(I), I=1,N)
     :             /-9.635265432D-6, 1D0, 1D0, -1D0/
      DATA (CTYPE(I), I=1,N)
     :           /'WAVE-F2W', 'XLAT-xxx ', 'TIME    ', 'XLON-xxx '/
      DATA (CRVAL(I), I=1,N)
     :             /0.214982042D0, -30D0, -2D3, 150D0/
      DATA LONPOLE /150D0/
      DATA LATPOLE /999D0/
      DATA RESTFRQ /1.42040575D9/
      DATA RESTWAV /0D0/

*     Set PVi_m cards for the longitude axis (I = 4) so that the
*     fiducial native coordinates are at the native pole, i.e.
*     (phi0,theta0) = (0,90), but without any fiducial offset.  We do
*     this as a test, and also so that all projections will be
*     exercised with the same obliquity parameters.
      DATA (PVI(K), PVM(K), PV(K), K=1,2)
     :             /4, 1,  0D0,
     :              4, 2, 90D0/

*     PVi_m cards for the latitude axis (I = 2).  Value may be reset
*     below.
      DATA (PVI(K), PVM(K), PV(K), K=3,4)
     :             /2, 1, 0D0,
     :              2, 2, 0D0/
*-----------------------------------------------------------------------
*     Uncomment the following two lines to raise SIGFPE on floating
*     point exceptions for the Sun FORTRAN compiler.  This signal can
*     be caught within 'dbx' by issuing the command "catch FPE".
*#include <floatingpoint.h>
*     call ieee_handler ('set', 'common', SIGFPE_ABORT)

      WRITE (6, 10)
 10   FORMAT (/,'Testing closure of WCSLIB world coordinate ',
     :          'transformation routines',/,
     :          '-------------------------------------------',
     :          '-----------------------')


*     ARC: zenithal/azimuthal equidistant.
      CTYPE(2)(6:8) = 'ARC'
      CTYPE(4)(6:8) = 'ARC'
      NPV = 2
      CALL WCSEX (TOL)

*     ZEA: zenithal/azimuthal equal area.
      CTYPE(2)(6:8) = 'ZEA'
      CTYPE(4)(6:8) = 'ZEA'
      NPV = 2
      CALL WCSEX (TOL)

*     CYP: cylindrical perspective.
      CTYPE(2)(6:8) = 'CYP'
      CTYPE(4)(6:8) = 'CYP'
      NPV = 4
      PV(3) = 3D0
      PV(4) = 0.8D0
      CALL WCSEX (TOL)

*     CEA: cylindrical equal area.
      CTYPE(2)(6:8) = 'CEA'
      CTYPE(4)(6:8) = 'CEA'
      NPV = 3
      PV(3) = 0.75D0
      CALL WCSEX (TOL)

*     CAR: Cartesian.
      CTYPE(2)(6:8) = 'CAR'
      CTYPE(4)(6:8) = 'CAR'
      NPV = 2
      CALL WCSEX (TOL)

*     SFL: Sanson-Flamsteed.
      CTYPE(2)(6:8) = 'SFL'
      CTYPE(4)(6:8) = 'SFL'
      NPV = 2
      CALL WCSEX (TOL)

*     PAR: parabolic.
      CTYPE(2)(6:8) = 'PAR'
      CTYPE(4)(6:8) = 'PAR'
      NPV = 2
      CALL WCSEX (TOL)

*     MOL: Mollweide's projection.
      CTYPE(2)(6:8) = 'MOL'
      CTYPE(4)(6:8) = 'MOL'
      NPV = 2
      CALL WCSEX (TOL)

*     AIT: Hammer-Aitoff.
      CTYPE(2)(6:8) = 'AIT'
      CTYPE(4)(6:8) = 'AIT'
      NPV = 2
      CALL WCSEX (TOL)

*     COE: conic equal area.
      CTYPE(2)(6:8) = 'COE'
      CTYPE(4)(6:8) = 'COE'
      NPV = 4
      PV(3) = 60D0
      PV(4) = 15D0
      CALL WCSEX (TOL)

*     COD: conic equidistant.
      CTYPE(2)(6:8) = 'COD'
      CTYPE(4)(6:8) = 'COD'
      NPV = 4
      PV(3) = 60D0
      PV(4) = 15D0
      CALL WCSEX (TOL)

*     BON: Bonne's projection.
      CTYPE(2)(6:8) = 'BON'
      CTYPE(4)(6:8) = 'BON'
      NPV = 3
      PV(3) = 30D0
      CALL WCSEX (TOL)

*     PCO: polyconic.
      CTYPE(2)(6:8) = 'PCO'
      CTYPE(4)(6:8) = 'PCO'
      NPV = 2
      CALL WCSEX (TOL)

*     TSC: tangential spherical cube.
      CTYPE(2)(6:8) = 'TSC'
      CTYPE(4)(6:8) = 'TSC'
      NPV = 2
      CALL WCSEX (TOL)

*     QSC: quadrilateralized spherical cube.
      CTYPE(1) = 'CUBEFACE'
      CTYPE(2)(6:8) = 'QSC'
      CTYPE(4)(6:8) = 'QSC'
      NPV = 2
      CALL WCSEX (TOL)

      END


*-----------------------------------------------------------------------
      SUBROUTINE WCSEX (TOL)
*-----------------------------------------------------------------------
*   WCSEX tests closure of WCSS2P and WCSP2S.
*
*   Given:
*      TOL      D        Reporting tolerance, degrees.
*-----------------------------------------------------------------------
      INTEGER   NELEM
      PARAMETER (NELEM = 9)

      INTEGER   K, LAT, LATIDX, LNGIDX, SPCIDX, STAT(0:360), STATUS
      DOUBLE PRECISION DLAT, DLATMX, DLNG, DLNGMX, FREQ,
     :          IMG1(NELEM,0:360), IMG2(NELEM,0:360), LAT1, LNG1,
     :          PHI1(0:360), PHI2(0:360), PIX(NELEM,0:360),
     :          THETA1(0:360), THETA2(0:360), TOL, WORLD(NELEM,0:360)
      CHARACTER PCODE*3

      INCLUDE 'wcs.inc'
      INCLUDE 'cel.inc'
      INCLUDE 'prj.inc'
      INTEGER   CEL(CELLEN), PRJ(PRJLEN), WCS(WCSLEN)
*-----------------------------------------------------------------------
*     This routine simulates the actions of a FITS header parser.
      CALL PARSER (WCS)

      STATUS = WCSGET (WCS, WCS_CEL, CEL)
      STATUS = CELGET (CEL, CEL_PRJ, PRJ)
      STATUS = PRJGET (PRJ, PRJ_CODE, PCODE)
      WRITE (6, 10) PCODE, TOL
 10   FORMAT (/,'Testing ',A,'; reporting tolerance',1PG8.1,' deg.')


*     Get indices.
      STATUS = WCSGET (WCS, WCS_LNG,  LNGIDX)
      STATUS = WCSGET (WCS, WCS_LAT,  LATIDX)
      STATUS = WCSGET (WCS, WCS_SPEC, SPCIDX)

*     Initialize non-celestial world coordinates.
      FREQ = 1.42040595D9 - 180D0 * 62500D0
      DO 20 K = 0, 360
         WORLD(1,K) = 0D0
         WORLD(2,K) = 0D0
         WORLD(3,K) = 0D0
         WORLD(4,K) = 0D0

         WORLD(SPCIDX,K) = 2.99792458D8 / FREQ
         FREQ = FREQ + 62500D0
 20   CONTINUE

      DLNGMX = 0D0
      DLATMX = 0D0
      DO 100 LAT = 90, -90, -1
         LAT1 = DBLE(LAT)

         LNG1 = -180D0
         DO 30 K = 0, 360
            WORLD(LNGIDX,K) = LNG1
            WORLD(LATIDX,K) = LAT1
            LNG1 = LNG1 + 1D0
 30      CONTINUE

         STATUS = WCSS2P (WCS, 361, NELEM, WORLD, PHI1, THETA1, IMG1,
     :                    PIX, STAT)
         IF (STATUS.NE.0) THEN
            WRITE (6, 40) PCODE, LAT1, STATUS
 40         FORMAT (3X,A3,': LAT1 =',F20.15, '  error',I3)
            GO TO 100
         END IF

         STATUS = WCSP2S (WCS, 361, NELEM, PIX, IMG2, PHI2, THETA2,
     :                    WORLD, STAT)
         IF (STATUS.NE.0) THEN
            WRITE (6, 50) PCODE, LAT1, STATUS
 50         FORMAT (3X,A3,': LAT1 =',F20.15,/, '  error',I3)
            GO TO 100
         END IF

         LNG1 = -180D0
         DO 90 K = 0, 360
            DLNG = ABS(WORLD(LNGIDX,K) - LNG1)
            IF (DLNG.GT.180D0) DLNG = ABS(DLNG - 360D0)
            IF (ABS(LAT).NE.90 .AND. DLNG.GT.DLNGMX) DLNGMX = DLNG

            DLAT = ABS(WORLD(LATIDX,K) - LAT1)
            IF (DLAT.GT.DLATMX) DLATMX = DLAT

            IF (DLAT.GT.TOL .OR.
     :         (DLNG.GT.TOL .AND. ABS(LAT).NE.90)) THEN
               WRITE (6, 60) PCODE, LNG1, LAT1, WORLD(LNGIDX,K),
     :            WORLD(LATIDX,K)
 60            FORMAT (8X,A3,': LNG1 =',F20.15,'    LAT1 =',F20.15,/,
     :                 8X,'     LNG2 =',F20.15,'    LAT2 =',F20.15)
               WRITE (6, 70) PHI1(K), THETA1(K), PHI2(K), THETA2(K)
 70            FORMAT (8X,'     PHI1 =',F20.15,'  THETA1 =',F20.15,/,
     :                 8X,'     PHI2 =',F20.15,'  THETA2 =',F20.15)
               WRITE (6, 80) IMG1(LNGIDX,K), IMG1(LATIDX,K),
     :                       IMG2(LNGIDX,K), IMG2(LATIDX,K)
 80            FORMAT (8X,'       X1 =',F20.15,'      Y1 =',F20.15,/,
     :                 8X,'       X2 =',F20.15,'      Y2 =',F20.15)
            END IF

            LNG1 = LNG1 + 1D0
 90      CONTINUE
 100  CONTINUE

      WRITE (6, 110) DLNGMX, DLATMX
 110  FORMAT ('     Maximum closure residual: lng',1P,G10.3,'   lat',
     :        G10.3)

      STATUS = WCSFREE(WCS)


      RETURN
      END

*-----------------------------------------------------------------------
      SUBROUTINE PARSER (WCS)
*-----------------------------------------------------------------------
* In practice a parser would read the FITS header until it encountered
* the NAXIS card which must occur near the start of the header, before
* any of the WCS keywords.  It would then use WCSINI to allocate memory
* for arrays in the WCSPRM "data structure" and set default values.
*
* In this simulation the header keyvalues are set in the main program in
* variables passed in COMMON.
*-----------------------------------------------------------------------
*     Number of axes.
      INTEGER   N
      PARAMETER (N = 4)

      INTEGER   I, J, K, NAXIS, NPV, PVI(4), PVM(4), STATUS, WCS(*)
      DOUBLE PRECISION CDELT(N), CRPIX(N), CRVAL(N), LATPOLE, LONPOLE,
     :          PC(N,N), PV(4), RESTFRQ, RESTWAV
      CHARACTER CTYPE(N)*72

      INCLUDE 'wcs.inc'

      COMMON /HEADER/ NAXIS, NPV, CRPIX, PC, CDELT, CRVAL, LONPOLE,
     :                LATPOLE, RESTFRQ, RESTWAV, PVI, PVM, PV
      COMMON /HEADCH/ CTYPE
*-----------------------------------------------------------------------
      STATUS = WCSINI (NAXIS, WCS)

      DO 20 I = 1, NAXIS
         STATUS = WCSPUT (WCS, WCS_CRPIX, CRPIX(I), I, 0)

         DO 10 J = 1, NAXIS
            STATUS = WCSPUT (WCS, WCS_PC, PC(I,J), I, J)
 10      CONTINUE

         STATUS = WCSPUT (WCS, WCS_CDELT, CDELT(I), I, 0)
         STATUS = WCSPUT (WCS, WCS_CTYPE, CTYPE(I), I, 0)
         STATUS = WCSPUT (WCS, WCS_CRVAL, CRVAL(I), I, 0)
 20   CONTINUE

      STATUS = WCSPUT (WCS, WCS_LONPOLE, LONPOLE, 0, 0)
      STATUS = WCSPUT (WCS, WCS_LATPOLE, LATPOLE, 0, 0)

      STATUS = WCSPUT (WCS, WCS_RESTFRQ, RESTFRQ, 0, 0)
      STATUS = WCSPUT (WCS, WCS_RESTWAV, RESTWAV, 0, 0)

      DO 30 K = 1, NPV
         STATUS = WCSPUT (WCS, WCS_PV, PV(K), PVI(K), PVM(K))
 30   CONTINUE

*     Extract information from the FITS header.
      STATUS = WCSSET (WCS)

      RETURN
      END
