/*============================================================================
*
*   WCSLIB 3.3 - an implementation of the FITS WCS convention.
*   Copyright (C) 1995-2003, Mark Calabretta
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation; either version 2 of the License, or (at your
*   option) any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You should have received a copy of the GNU General Public License along
*   with this library; if not, write to the Free Software Foundation, Inc.,
*   675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta
*                      Australia Telescope National Facility, CSIRO
*                      PO Box 76
*                      Epping NSW 1710
*                      AUSTRALIA
*
*=============================================================================
*
*   twcs1 tests wcss2p() and wcsp2s() for closure on the 1 degree celestial
*   graticule for a number of selected projections.
*
*   $Id: twcs1.c,v 3.3 2003/10/21 08:10:05 mcalabre Exp $
*---------------------------------------------------------------------------*/

#include <malloc.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <wcs.h>

#ifndef __STDC__
#ifndef const
#define const
#endif
#endif

void wcsex();
void parser(struct wcsprm *);

/* Reporting tolerance for wcsex(). */
const double tol = 1.0e-9;


/* In real life these would be encoded as FITS header cards. */
const int NAXIS = 4;
const double CRPIX[4] =  {  513.0,  0.0,  0.0,  0.0};
const double PC[4][4] = {{    1.1,  0.0,  0.0,  0.0},
                         {    0.0,  1.0,  0.0,  0.1},
                         {    0.0,  0.0,  1.0,  0.0},
                         {    0.0,  0.2,  0.0,  1.0}};
const double CDELT[4] =  {-9.635265432e-6, 1.0, 1.0, -1.0};

				/* The "xxx" is reset in main(). */
char CTYPE[4][9] = {"WAVE-F2W", "XLAT-xxx", "TIME    ", "XLON-xxx"};

const double CRVAL[4] = {0.214982042, -30.0, -2e3, 150.0};
const double LONPOLE  = 150.0;
const double LATPOLE  = 999.0;
const double RESTFRQ  =   1.42040575e9;
const double RESTWAV  =   0.0;

int NPV;
struct pvcard PV[10];		/* Projection parameters are set in main(). */


int main()

{
   register int j;

   /* Uncomment the following two lines to raise SIGFPE on floating point
    * exceptions for the Sun FORTRAN compiler.  This signal can be caught
    * within 'dbx' by issuing the command "catch FPE".
    */
/* #include <floatingpoint.h> */
/* ieee_handler("set", "common", SIGFPE_ABORT); */


   printf("\nTesting closure of WCSLIB world coordinate transformation "
          "routines\n--------------------------------------------------"
          "----------------\n");

   /* List error messages. */
   printf("\nList of wcs error codes:\n");
   for (j = 1; j <= 11 ; j++) {
      printf("   %2d: %s.\n", j, wcs_errmsg[j]);
   }


   /*----------------------------------------------------------*/
   /* Set the PVi_m cards for the longitude axis so that the   */
   /* fiducial native coordinates are at the native pole, i.e. */
   /* (phi0,theta0) = (0,90), but without any fiducial offset. */
   /* We do this as a test, and also so that all projections   */
   /* will be exercised with the same obliquity parameters.    */
   /*----------------------------------------------------------*/
   PV[0].i = 4;			/* Longitude is on axis 4.     */
   PV[0].m = 1;			/* Parameter number 1.         */
   PV[0].value =  0.0;		/* Fiducial native longitude.  */

   PV[1].i = 4;			/* Longitude is on axis 4.     */
   PV[1].m = 2;			/* Parameter number 2.         */
   PV[1].value = 90.0;		/* Fiducial native latitude.   */

   /* Set the PVi_m cards for the latitude axis.               */
   PV[2].i = 2;			/* Latitude is on axis 2.      */
   PV[2].m = 1;			/* Parameter number 1.         */
   PV[2].value = 0.0;		/* PVi_1 (set below).          */

   PV[3].i = 2;			/* Latitude is on axis 2.      */
   PV[3].m = 2;			/* Parameter number 2.         */
   PV[3].value = 0.0;		/* PVi_2 (set below).          */

   /* ARC: zenithal/azimuthal equidistant. */
   strncpy(&CTYPE[1][5], "ARC", 3);
   strncpy(&CTYPE[3][5], "ARC", 3);
   NPV = 2;
   wcsex();

   /* ZEA: zenithal/azimuthal equal area. */
   strncpy(&CTYPE[1][5], "ZEA", 3);
   strncpy(&CTYPE[3][5], "ZEA", 3);
   NPV = 2;
   wcsex();

   /* CYP: cylindrical perspective. */
   strncpy(&CTYPE[1][5], "CYP", 3);
   strncpy(&CTYPE[3][5], "CYP", 3);
   NPV = 4;
   PV[2].value = 3.0;
   PV[3].value = 0.8;
   wcsex();

   /* CEA: cylindrical equal area. */
   strncpy(&CTYPE[1][5], "CEA", 3);
   strncpy(&CTYPE[3][5], "CEA", 3);
   NPV = 3;
   PV[2].value = 0.75;
   wcsex();

   /* CAR: plate carree. */
   strncpy(&CTYPE[1][5], "CAR", 3);
   strncpy(&CTYPE[3][5], "CAR", 3);
   NPV = 2;
   wcsex();

   /* SFL: Sanson-Flamsteed. */
   strncpy(&CTYPE[1][5], "SFL", 3);
   strncpy(&CTYPE[3][5], "SFL", 3);
   NPV = 2;
   wcsex();

   /* PAR: parabolic. */
   strncpy(&CTYPE[1][5], "PAR", 3);
   strncpy(&CTYPE[3][5], "PAR", 3);
   NPV = 2;
   wcsex();

   /* MOL: Mollweide's projection. */
   strncpy(&CTYPE[1][5], "MOL", 3);
   strncpy(&CTYPE[3][5], "MOL", 3);
   NPV = 2;
   wcsex();

   /* AIT: Hammer-Aitoff. */
   strncpy(&CTYPE[1][5], "AIT", 3);
   strncpy(&CTYPE[3][5], "AIT", 3);
   NPV = 2;
   wcsex();

   /* COE: conic equal area. */
   strncpy(&CTYPE[1][5], "COE", 3);
   strncpy(&CTYPE[3][5], "COE", 3);
   NPV = 4;
   PV[2].value = 60.0;
   PV[3].value = 15.0;
   wcsex();

   /* COD: conic equidistant. */
   strncpy(&CTYPE[1][5], "COD", 3);
   strncpy(&CTYPE[3][5], "COD", 3);
   NPV = 4;
   PV[2].value = 60.0;
   PV[3].value = 15.0;
   wcsex();

   /* BON: Bonne's projection. */
   strncpy(&CTYPE[1][5], "BON", 3);
   strncpy(&CTYPE[3][5], "BON", 3);
   NPV = 3;
   PV[2].value = 30.0;
   wcsex();

   /* PCO: polyconic. */
   strncpy(&CTYPE[1][5], "PCO", 3);
   strncpy(&CTYPE[3][5], "PCO", 3);
   NPV = 2;
   wcsex();

   /* TSC: tangential spherical cube. */
   strncpy(&CTYPE[1][5], "TSC", 3);
   strncpy(&CTYPE[3][5], "TSC", 3);
   NPV = 2;
   wcsex();

   /* QSC: quadrilateralized spherical cube. */
   strncpy(&CTYPE[1][5], "QSC", 3);
   strncpy(&CTYPE[3][5], "QSC", 3);
   NPV = 2;
   wcsex();

   return 0;
}


/*----------------------------------------------------------------------------
*   wcsex() tests closure of wcss2p() and wcsp2s().
*---------------------------------------------------------------------------*/

void wcsex()

{
#define NELEM 9

   int k, lat, lng, stat[361], status;
   double dlat, dlatmx, dlng, dlngmx, freq, img1[361][NELEM],
          img2[361][NELEM], lat1, lng1, phi1[361], phi2[361], pix[361][NELEM],
          theta1[361], theta2[361], world[361][NELEM];
   struct wcsprm *wcs;


   /* The following routine simulates the actions of a FITS header parser. */
   wcs = (struct wcsprm *)malloc(sizeof(struct wcsprm));
   wcs->flag = -1;
   parser(wcs);

   printf("\nTesting %s; reporting tolerance %5.1g deg.\n", wcs->cel.prj.code,
      tol);


   /* Initialize non-celestial world coordinates. */
   freq = 1.42040595e9 - 180.0 * 62500.0;
   for (k = 0; k < 361; k++) {
      world[k][0] = 0.0;
      world[k][1] = 0.0;
      world[k][2] = 0.0;
      world[k][3] = 0.0;

      world[k][wcs->spec] = 2.99792458e8 / freq;
      freq += 62500.0;
   }


   dlngmx = 0.0;
   dlatmx = 0.0;
   for (lat = 90; lat >= -90; lat--) {
      lat1 = (double)lat;

      for (lng = -180, k = 0; lng <= 180; lng++, k++) {
         lng1 = (double)lng;

         world[k][wcs->lng] = lng1;
         world[k][wcs->lat] = lat1;
      }

      if (status = wcss2p(wcs, 361, NELEM, world[0], phi1, theta1, img1[0],
                       pix[0], stat)) {
         printf("   %3s(s2x): lat1 =%20.15f  error %3d\n",
            wcs->cel.prj.code, lat1, status);
         continue;
      }

      if (status = wcsp2s(wcs, 361, NELEM, pix[0], img2[0], phi2, theta2,
                       world[0], stat)) {
         printf("   %3s(x2s): lat1 =%20.15f  error%3d\n",
            wcs->cel.prj.code, lat1, status);
         continue;
      }

      for (lng = -180, k = 0; lng <= 180; lng++, k++) {
         lng1 = (double)lng;

         dlng = fabs(world[k][wcs->lng] - lng1);
         if (dlng > 180.0) dlng = fabs(dlng - 360.0);
         if (abs(lat) != 90 && dlng > dlngmx) dlngmx = dlng;

         dlat = fabs(world[k][wcs->lat] - lat1);
         if (dlat > dlatmx) dlatmx = dlat;

         if (dlat > tol || (dlng > tol && abs(lat) != 90)) {
            printf("        %3s: lng1 =%20.15f    lat1 =%20.15f\n"
                   "             lng2 =%20.15f    lat2 =%20.15f\n",
               wcs->cel.prj.code, lng1, lat1, world[k][wcs->lng],
               world[k][wcs->lat]);
            printf("             phi1 =%20.15f  theta1 =%20.15f\n"
                   "             phi2 =%20.15f  theta2 =%20.15f\n",
               phi1[k], theta1[k], phi2[k], theta2[k]);
            printf("               x1 =%20.15f      y1 =%20.15f\n"
                   "               x2 =%20.15f      y2 =%20.15f\n",
               img1[k][wcs->lng], img1[k][wcs->lat], img2[k][wcs->lng],
               img2[k][wcs->lat]);
         }
      }
   }

   printf("     Maximum closure residual: lng%10.3e   lat%10.3e\n",
      dlngmx, dlatmx);

   wcsfree(wcs);
   free(wcs);
}

/*--------------------------------------------------------------------------*/

void parser(wcs)

struct wcsprm *wcs;

{
   int i, j, status;
   double *pcij;

   /* In practice a parser would read the FITS header until it encountered  */
   /* the NAXIS card which must occur near the start of the header before   */
   /* any of the WCS keywords.  It would then use wcsini() to allocate      */
   /* memory for arrays in the wcsprm struct and set default values.        */
   /* In this simulation the header keyvalues are set as global variables.  */
   wcsini(1, NAXIS, wcs);


   /* Now the parser scans the FITS header, identifying WCS keywords and    */
   /* loading their values into the appropriate elements of the wcsprm      */
   /* struct.                                                               */

   for (j = 0; j < NAXIS; j++) {
      wcs->crpix[j] = CRPIX[j];
   }

   pcij = wcs->pc;
   for (i = 0; i < NAXIS; i++) {
      for (j = 0; j < NAXIS; j++) {
         *(pcij++) = PC[i][j];
      }
   }

   for (i = 0; i < NAXIS; i++) {
      wcs->cdelt[i] = CDELT[i];
   }

   for (i = 0; i < NAXIS; i++) {
      strcpy(wcs->ctype[i], &CTYPE[i][0]);
   }

   for (i = 0; i < NAXIS; i++) {
      wcs->crval[i] = CRVAL[i];
   }

   wcs->lonpole = LONPOLE;
   wcs->latpole = LATPOLE;

   wcs->restfrq = RESTFRQ;
   wcs->restwav = RESTWAV;

   wcs->npv = NPV;
   for (i = 0; i < NPV; i++) {
      wcs->pv[i] = PV[i];
   }

   /* Extract information from the FITS header. */
   if (status = wcsset(wcs)) {
      printf("wcsset error%3d\n", status);
   }

   return;
}
