/*============================================================================
*
*   WCSLIB 3.3 - an implementation of the FITS WCS convention.
*   Copyright (C) 1995-2003, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or modify it
*   under the terms of the GNU Library General Public License as published
*   by the Free Software Foundation; either version 2 of the License, or (at
*   your option) any later version.
*
*   This library is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library
*   General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public License
*   along with this library; if not, write to the Free Software Foundation,
*   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta
*                      Australia Telescope National Facility, CSIRO
*                      PO Box 76
*                      Epping NSW 1710
*                      AUSTRALIA
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: wcs_f.c,v 3.3 2003/10/21 08:44:04 mcalabre Exp $
*===========================================================================*/

#include <string.h>
#include <wcs.h>

#define WCS_FLAG     100
#define WCS_NAXIS    101
#define WCS_CRPIX    102
#define WCS_PC       103
#define WCS_CDELT    104
#define WCS_CTYPE    105
#define WCS_CUNIT    106
#define WCS_CRVAL    107
#define WCS_LONPOLE  108
#define WCS_LATPOLE  109
#define WCS_RESTFRQ  110
#define WCS_RESTWAV  111
#define WCS_NPV      112
#define WCS_PV       113
#define WCS_ALTLIN   114
#define WCS_CD       115
#define WCS_CROTA    116

#define WCS_LNGTYP   200
#define WCS_LATTYP   201
#define WCS_LNG      202
#define WCS_LAT      203
#define WCS_SPEC     204
#define WCS_CUBEFACE 205
#define WCS_LIN      206
#define WCS_CEL      207
#define WCS_SPC      208

/*--------------------------------------------------------------------------*/

int wcsini_(naxis, wcs)

const int *naxis;
int *wcs;

{
   return wcsini(1, *naxis, (struct wcsprm *)wcs);
}

/*--------------------------------------------------------------------------*/

int wcscopy_(wcssrc, wcsdst)

const int *wcssrc;
int *wcsdst;

{
   return wcscopy(1, (const struct wcsprm *)wcssrc, (struct wcsprm *)wcsdst);
}

/*--------------------------------------------------------------------------*/

int wcsput_(wcs, what, value, i, j)

int *wcs;
const int *what;
const void *value;
const int *i, *j;

{
   int i0, j0, k;
   const char *cvalp;
   const int  *ivalp;
   const double *dvalp;
   struct wcsprm *wcsp;

   /* Cast pointers. */
   wcsp  = (struct wcsprm *)wcs;
   cvalp = (const char *)value;
   ivalp = (const int *)value;
   dvalp = (const double *)value;

   /* Convert 1-relative FITS axis numbers to 0-relative C array indices. */
   i0 = *i - 1;
   j0 = *j - 1;

   wcsp->flag = 0;

   switch (*what) {
   case WCS_FLAG:
      wcsp->flag = *ivalp;
      break;
   case WCS_NAXIS:
      wcsp->naxis = *ivalp;
      break;
   case WCS_CRPIX:
      wcsp->crpix[i0] = *dvalp;
      break;
   case WCS_PC:
      k = (i0)*(wcsp->naxis) + (j0);
      *(wcsp->pc+k) = *dvalp;
      break;
   case WCS_CDELT:
      wcsp->cdelt[i0] = *dvalp;
      break;
   case WCS_CTYPE:
      strncpy(wcsp->ctype[i0], cvalp, 72);
      wcsp->ctype[i0][71] = '\0';
      break;
   case WCS_CUNIT:
      strncpy(wcsp->cunit[i0], cvalp, 72);
      wcsp->cunit[i0][71] = '\0';
      break;
   case WCS_CRVAL:
      wcsp->crval[i0] = *dvalp;
      break;
   case WCS_LONPOLE:
      wcsp->lonpole = *dvalp;
      break;
   case WCS_LATPOLE:
      wcsp->latpole = *dvalp;
      break;
   case WCS_RESTFRQ:
      wcsp->restfrq = *dvalp;
      break;
   case WCS_RESTWAV:
      wcsp->restwav = *dvalp;
      break;
   case WCS_NPV:
      return 1;
      break;
   case WCS_PV:
      (wcsp->pv + wcsp->npv)->i = *i;
      (wcsp->pv + wcsp->npv)->m = *j;
      (wcsp->pv + wcsp->npv)->value = *dvalp;
      (wcsp->npv)++;
      break;
   case WCS_ALTLIN:
      wcsp->altlin = *ivalp;
      break;
   case WCS_CD:
      k = (i0)*(wcsp->naxis) + (j0);
      *(wcsp->cd+k) = *dvalp;
      break;
   case WCS_CROTA:
      wcsp->crota[i0] = *dvalp;
      break;
   default:
      return 1;
   }

   return 0;
}

/*--------------------------------------------------------------------------*/

int wcsget_(wcs, what, value)

const int *wcs;
const int *what;
void *value;

{
   int i, j, k, naxis;
   char   *cvalp;
   int    *ivalp;
   double *dvalp;
   const int    *iwcsp;
   const double *dwcsp;
   const struct wcsprm *wcsp;

   /* Cast pointers. */
   wcsp  = (const struct wcsprm *)wcs;
   cvalp = (char *)value;
   ivalp = (int *)value;
   dvalp = (double *)value;

   naxis = wcsp->naxis;

   switch (*what) {
   case WCS_FLAG:
      *ivalp = wcsp->flag;
      break;
   case WCS_NAXIS:
      *ivalp = naxis;
      break;
   case WCS_CRPIX:
      for (i = 0; i < naxis; i++) {
         *(dvalp++) = wcsp->crpix[i];
      }
      break;
   case WCS_PC:
      /* C row-major to FORTRAN column-major. */
      for (j = 0; j < naxis; j++) {
         dwcsp = wcsp->pc + j;
         for (i = 0; i < naxis; i++) {
            *(dvalp++) = *dwcsp;
            dwcsp += naxis;
         }
      }
      break;
   case WCS_CDELT:
      for (i = 0; i < naxis; i++) {
         *(dvalp++) = wcsp->cdelt[i];
      }
      break;
   case WCS_CTYPE:
      for (i = 0; i < naxis; i++) {
         strncpy(cvalp, wcsp->ctype[i], 72);
         cvalp += 72;
      }
      break;
   case WCS_CUNIT:
      for (i = 0; i < naxis; i++) {
         strncpy(cvalp, wcsp->cunit[i], 72);
         cvalp += 72;
      }
      break;
   case WCS_CRVAL:
      for (i = 0; i < naxis; i++) {
         *(dvalp++) = wcsp->crval[i];
      }
      break;
   case WCS_LONPOLE:
      *dvalp = wcsp->lonpole;
      break;
   case WCS_LATPOLE:
      *dvalp = wcsp->latpole;
      break;
   case WCS_RESTFRQ:
      *dvalp = wcsp->restfrq;
      break;
   case WCS_RESTWAV:
      *dvalp = wcsp->restwav;
      break;
   case WCS_NPV:
      *ivalp = wcsp->npv;
      break;
   case WCS_PV:
      for (k = 0; k < wcsp->npv; k++) {
         *(dvalp++) = (wcsp->pv + k)->i;
         *(dvalp++) = (wcsp->pv + k)->m;
         *(dvalp++) = (wcsp->pv + k)->value;
      }
      break;
   case WCS_ALTLIN:
      *ivalp = wcsp->altlin;
      break;
   case WCS_CD:
      /* C row-major to FORTRAN column-major. */
      for (j = 0; j < naxis; j++) {
         dwcsp = wcsp->cd + j;
         for (i = 0; i < naxis; i++) {
            *(dvalp++) = *dwcsp;
            dwcsp += naxis;
         }
      }
      break;
   case WCS_CROTA:
      for (i = 0; i < naxis; i++) {
         *(dvalp++) = wcsp->crota[i];
      }
      break;
   case WCS_LNGTYP:
      strncpy(cvalp, wcsp->lngtyp, 4);
      break;
   case WCS_LATTYP:
      strncpy(cvalp, wcsp->lattyp, 4);
      break;
   case WCS_LNG:
      *ivalp = wcsp->lng + 1;
      break;
   case WCS_LAT:
      *ivalp = wcsp->lat + 1;
      break;
   case WCS_SPEC:
      *ivalp = wcsp->spec + 1;
      break;
   case WCS_CUBEFACE:
      *ivalp = wcsp->cubeface;
      break;
   case WCS_LIN:
      k = (int *)(&(wcsp->lin)) - (int *)wcsp;
      iwcsp = wcs + k;
      for (k = 0; k < LINLEN; k++) {
         *(ivalp++) = *(iwcsp++);
      }
      break;
   case WCS_CEL:
      k = (int *)(&(wcsp->cel)) - (int *)wcsp;
      iwcsp = wcs + k;
      for (k = 0; k < CELLEN; k++) {
         *(ivalp++) = *(iwcsp++);
      }
      break;
   case WCS_SPC:
      k = (int *)(&(wcsp->spc)) - (int *)wcsp;
      iwcsp = wcs + k;
      for (k = 0; k < SPCLEN; k++) {
         *(ivalp++) = *(iwcsp++);
      }
      break;
   default:
      return 1;
   }

   return 0;
}

/*--------------------------------------------------------------------------*/

int wcsfree_(wcs)

int *wcs;

{
   return wcsfree((struct wcsprm *)wcs);
}

/*--------------------------------------------------------------------------*/

int wcsprt_(wcs)

int *wcs;

{
   return wcsprt((struct wcsprm *)wcs);
}

/*--------------------------------------------------------------------------*/

int wcsset_(wcs)

int *wcs;

{
   return wcsset((struct wcsprm *)wcs);
}

/*--------------------------------------------------------------------------*/

int wcsp2s_(wcs, ncoord, nelem, pixcrd, imgcrd, phi, theta, world, stat)

int *wcs;
const int *ncoord, *nelem;
const double pixcrd[];
double imgcrd[];
double phi[], theta[];
double world[];
int stat[];

{
   return wcsp2s((struct wcsprm *)wcs, *ncoord, *nelem, pixcrd, imgcrd, phi,
                  theta, world, stat);
}

/*--------------------------------------------------------------------------*/

int wcss2p_(wcs, ncoord, nelem, world, phi, theta, imgcrd, pixcrd, stat)

int* wcs;
const int *ncoord, *nelem;
const double world[];
double phi[], theta[];
double imgcrd[];
double pixcrd[];
int stat[];

{
   return wcss2p((struct wcsprm *)wcs, *ncoord, *nelem, world, phi, theta,
                  imgcrd, pixcrd, stat);
}

/*--------------------------------------------------------------------------*/

int wcsmix_(wcs, mixpix, mixcel, vspan, vstep, viter, world, phi, theta,
            imgcrd, pixcrd)

int *wcs;
const int *mixpix, *mixcel;
const double vspan[2], *vstep;
int *viter;
double world[];
double phi[], theta[];
double imgcrd[];
double pixcrd[];

{
   return wcsmix((struct wcsprm *)wcs, *mixpix-1, *mixcel, vspan, *vstep,
                 *viter, world, phi, theta, imgcrd, pixcrd);
}
