/*============================================================================
*
*   WCSLIB 4.0 - an implementation of the FITS WCS convention.
*   Copyright (C) 1995-2005, Mark Calabretta
*
*   WCSLIB is free software; you can redistribute it and/or modify it under
*   the terms of the GNU General Public License as published by the Free
*   Software Foundation; either version 2 of the License, or (at your option)
*   any later version.
*
*   WCSLIB is distributed in the hope that it will be useful, but WITHOUT ANY
*   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
*   FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
*   details.
*
*   You should have received a copy of the GNU General Public License along
*   with WCSLIB; if not, write to the Free Software Foundation, Inc.,
*   59 Temple Place, Suite 330, Boston, MA  02111-1307, USA
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta
*                      Australia Telescope National Facility, CSIRO
*                      PO Box 76
*                      Epping NSW 1710
*                      AUSTRALIA
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   http://www.atnf.csiro.au/~mcalabre/index.html
*   $Id: ttab2.c,v 4.0 2005/02/04 06:50:22 cal103 Exp $
*=============================================================================
*
*   ttab2 tests the -TAB routines using PGPLOT for graphical display.  It
*   demonstrates the nature of linear interpolation in 2 dimensions by
*   contouring the interior a single 2 x 2 interpolation element as the
*   values in each corner change.
*
*---------------------------------------------------------------------------*/

#include <cpgplot.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include <tab.h>

int main()

{
   /* Set up a 2 x 2 lookup table. */
   const int M  = 2;
   const int K1 = 2;
   const int K2 = 2;
   const int K[] = {K1, K2};
   const int map[] = {0, 1};
   const double crval[] = {0.0, 0.0};

   /* Number of subdivisions on each side of the interpolation element. */
   const int NP = 128;

   char text[80];
   int i, j, k, l, l1, l2, l3, m, stat[NP*NP], status;
   float array[NP][NP], clev[11];
   const float scl = 1.0f/(NP-1);
   float ltm[] = {-scl, scl, 0.0f, -scl, 0.0f, scl};
   double x[NP][NP][M], world[NP][NP][M];
   struct tabprm tab;

   printf("\nTesting WCSLIB coordinate lookup table routines\n");
   printf("-------------------------------------------------\n");

   /* List status return messages. */
   printf("\nList of tab status return values:\n");
   for (j = 1; j <= 4 ; j++) {
      printf("   %d: %s.\n", j, tab_errmsg[j]);
   }

   printf("\n");


   /* PGPLOT initialization. */
   strcpy(text, "/xwindow");
   cpgbeg(0, text, 1, 1);
   cpgvstd();
   cpgsch(0.7f);

   /* The viewport is slightly oversized. */
   cpgwnad(-0.1f, 1.1f, -0.1f, 1.1f);

   for (l = 0; l <= 10; l++) {
      clev[l] = 0.1f*l;
   }


   /* Set up the lookup table. */
   tab.flag = -1;
   if (status = tabini(1, M, K, &tab)) {
      printf("tabini: %s.\n", tab_errmsg[status]);
      return 1;
   }

   tab.M = M;
   for (m = 0; m < tab.M; m++) {
      tab.K[m] = K[m];
      tab.map[m] = map[m];
      tab.crval[m] = crval[m];

      for (k = 0; k < tab.K[m]; k++) {
         tab.index[m][k] = (double)k;
      }
   }

   /* Subdivide the interpolation element. */
   for (i = 0; i < NP; i++) {
      for (j = 0; j < NP; j++) {
         x[i][j][0] = j*(K1-1.0)/(NP-1.0) - crval[0];
         x[i][j][1] = i*(K2-1.0)/(NP-1.0) - crval[1];
      }
   }

   /* The first coordinate element is static. */
   tab.coord[0] = 0.0;
   tab.coord[2] = 0.0;
   tab.coord[4] = 0.0;
   tab.coord[6] = 0.0;

   /* (k1,k2) = (0,0). */
   tab.coord[1] = 0.0;

   /* The second coordinate element varies in three of the corners. */
   for (l3 = 0; l3 <= 100; l3 += 20) {
      /* (k1,k2) = (1,1). */
      tab.coord[7] = 0.01 * l3;

      for (l1 = 0; l1 <= 100; l1 += 20) {
         /* (k1,k2) = (0,1). */
         tab.coord[3] = 0.01 * l1;

         cpgpage();
         for (l2 = 0; l2 <= 100; l2 += 2) {
            /* (k1,k2) = (1,0). */
            tab.coord[5] = 0.01 * l2;

            /* Compute coordinates within the interpolation element. */
            if (status = tabx2s(&tab, NP*NP, 2, (double *)x, (double *)world,
                                stat)) {
               printf("tabx2s: %s.\n", tab_errmsg[status]);
            }

            /* Start a new plot. */
            cpgbbuf();
            cpgeras();
            cpgsci(1);
            cpgbox("BCNST", 0.0f, 0, "BCNSTV", 0.0f, 0);

            /* Draw the boundary of the interpolation element in red. */
            cpgsci(2);
            cpgmove(0.0f, 0.0f);
            cpgdraw(1.0f, 0.0f);
            cpgdraw(1.0f, 1.0f);
            cpgdraw(0.0f, 1.0f);
            cpgdraw(0.0f, 0.0f);

            /* Label the value of the coordinate element in each corner. */
            sprintf(text, "%.1f", tab.coord[1]);
            cpgtext(-0.05f, -0.03f, text);
            sprintf(text, "%.1f", tab.coord[3]);
            cpgtext( 1.01f, -0.03f, text);
            sprintf(text, "%.2f", tab.coord[5]);
            cpgtext(-0.05f,  1.01f, text);
            sprintf(text, "%.1f", tab.coord[7]);
            cpgtext( 1.01f,  1.01f, text);

            cpgsci(5);
            /* Contour labelling: bottom. */
            for (l = 10; l < l1; l += 10) {
               sprintf(text, "%.1f", l*0.01f);
               cpgtext(((float)l)/l1 - 0.02f, -0.03f, text);
            }

            /* Contour labelling: left. */
            for (l = 10; l < l2; l += 10) {
               sprintf(text, "%.1f", l*0.01f);
               cpgtext(-0.05f, ((float)l)/l2 - 0.01f, text);
            }

            /* Contour labelling: right. */
            for (l = 10; l < 100; l += 10) {
               if (l <= l1 && l <= l3) continue;
               if (l >= l1 && l >= l3) continue;
               sprintf(text, "%.1f", l*0.01f);
               cpgtext(1.01f, ((float)(l-l1))/(l3-l1) - 0.01f, text);
            }

            /* Contour labelling: top. */
            for (l = 10; l < 100; l += 10) {
               if (l <= l2 && l <= l3) continue;
               if (l >= l2 && l >= l3) continue;
               sprintf(text, "%.1f", l*0.01f);
               cpgtext(((float)(l-l2))/(l3-l2) - 0.02f, 1.01f, text);
            }

            /* Draw contours for the second coordinate element. */
            for (i = 0; i < NP; i++) {
               for (j = 0; j < NP; j++) {
                  array[i][j] = world[i][j][1];
               }
            }

            cpgsci(m?5:1);
            cpgcont(array[0], NP, NP, 1, NP, 1, NP, clev, 11, ltm);

            cpgebuf();
         }
      }
   }

   cpgend();

   return 0;
}
