/*============================================================================

  WCSLIB 4.4 - an implementation of the FITS WCS standard.
  Copyright (C) 1995-2009, Mark Calabretta

  This file is part of WCSLIB.

  WCSLIB is free software: you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free
  Software Foundation, either version 3 of the License, or (at your option)
  any later version.

  WCSLIB is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
  more details.

  You should have received a copy of the GNU Lesser General Public License
  along with WCSLIB.  If not, see <http://www.gnu.org/licenses/>.

  Correspondence concerning WCSLIB may be directed to:
    Internet email: mcalabre@atnf.csiro.au
    Postal address: Dr. Mark Calabretta
                    Australia Telescope National Facility, CSIRO
                    PO Box 76
                    Epping NSW 1710
                    AUSTRALIA

  Author: Mark Calabretta, Australia Telescope National Facility
  http://www.atnf.csiro.au/~mcalabre/index.html
  $Id: twcs.c,v 4.4.1.1 2009/08/10 08:54:26 cal103 Exp cal103 $
*=============================================================================
*
* twcs tests wcss2p() and wcsp2s() for closure on an oblique 2-D slice through
* a 4-D image with celestial, spectral and logarithmic coordinate axes.
*
*---------------------------------------------------------------------------*/

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <wcslib.h>


void parser(struct wcsprm *);

/* Reporting tolerance. */
const double tol = 1.0e-10;


/* In real life these would be encoded as FITS header keyrecords. */
const int NAXIS = 2;
const double CRPIX[2] =  {  513.0,  0.0};
const double PC[2][2] = {{    1.1,  0.0},
                         {    0.0,  1.0}};
const double CDELT[4] =  {-1.0, 1.0};

char CTYPE[4][9] = {"XLON-CAR", "XLAT-CAR"};

const double CRVAL[4] = {0.0, 0.0};
const double LONPOLE  = 999.0;
const double LATPOLE  = 999.0;


int main()

{
  int    k, stat[3], status;
  double img[3][2], phi[3], pixel[3][2], theta[3], world[3][2];
  struct wcsprm *wcs;


  /* The following routine simulates the actions of a FITS header parser. */
  wcs = malloc(sizeof(struct wcsprm));
  wcs->flag = -1;
  parser(wcs);

  printf("\nReporting tolerance %5.1g pixel.\n", tol);

  world[0][wcs->lng] = 35.0;
  world[1][wcs->lng] = 35.0;
  world[2][wcs->lng] = 35.0;

  world[0][wcs->lat] =  90.0 + 1e-10;
  world[1][wcs->lat] =   0.0;
  world[2][wcs->lat] = -90.0;

  if ((status = wcss2p(wcs, 3, 2, world[0], phi, theta, img[0],
                       pixel[0], stat))) {
    printf("   wcss2p(1) ERROR %2d\n", status);
  }

  for (k = 0; k < 3; k++) {
    printf("%9.4f%9.4f  %9.4f%9.4f\n", world[k][0], world[k][1], phi[k],
      theta[k]);
  }

  wcsfree(wcs);
  free(wcs);

  return 0;
}

/*--------------------------------------------------------------------------*/

void parser(wcs)

struct wcsprm *wcs;

{
  int i, j, status;
  double *pcij;

  /* In practice a parser would read the FITS header until it encountered  */
  /* the NAXIS keyword which must occur near the start, before any of the  */
  /* WCS keywords.  It would then use wcsini() to allocate memory for      */
  /* arrays in the wcsprm struct and set default values.  In this          */
  /* simulation the header keyvalues are set as global variables.          */
  wcsini(1, NAXIS, wcs);


  /* Now the parser scans the FITS header, identifying WCS keywords and    */
  /* loading their values into the appropriate elements of the wcsprm      */
  /* struct.                                                               */

  for (j = 0; j < NAXIS; j++) {
    wcs->crpix[j] = CRPIX[j];
  }

  pcij = wcs->pc;
  for (i = 0; i < NAXIS; i++) {
    for (j = 0; j < NAXIS; j++) {
      *(pcij++) = PC[i][j];
    }
  }

  for (i = 0; i < NAXIS; i++) {
    wcs->cdelt[i] = CDELT[i];
  }

  for (i = 0; i < NAXIS; i++) {
    strcpy(wcs->ctype[i], &CTYPE[i][0]);
  }

  for (i = 0; i < NAXIS; i++) {
    wcs->crval[i] = CRVAL[i];
  }

  wcs->lonpole = LONPOLE;
  wcs->latpole = LATPOLE;

  /* Extract information from the FITS header. */
  if ((status = wcsset(wcs))) {
    printf("wcsset ERROR%3d\n", status);
  }

  return;
}
