/*============================================================================

  WCSLIB 5.0 - an implementation of the FITS WCS standard.
  Copyright (C) 1995-2015, Mark Calabretta

  This file is part of WCSLIB.

  WCSLIB is free software: you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free
  Software Foundation, either version 3 of the License, or (at your option)
  any later version.

  WCSLIB is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
  more details.

  You should have received a copy of the GNU Lesser General Public License
  along with WCSLIB.  If not, see http://www.gnu.org/licenses.

  Direct correspondence concerning WCSLIB to mark@calabretta.id.au

  Author: Mark Calabretta, Australia Telescope National Facility, CSIRO.
  http://www.atnf.csiro.au/people/Mark.Calabretta
  $Id: tdis1.c,v 5.0 2015/04/05 12:25:01 mcalabre Exp $
*=============================================================================
*
* tdis1 tests the WCSLIB distortion functions for closure.  Input comes from
* the* FITS file specified as an argument, or else from TPV7.fits.  The test
* is done via linp2x() and linx2p().
*
*---------------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <wcserr.h>
#include <wcshdr.h>
#include <wcsprintf.h>
#include <lin.h>
#include <dis.h>

#define HEADER_SIZE 36000

/* Absolute and fractional tolerance.  Distortions are typically used on  */
/* large images, so the absolute tolerance in the corners may not be very */
/* high, simply due to floating point precision.                          */
const double ATOL = 1.0e-9;
const double FTOL = 1.0e-10;


int main(int argc, char *argv[])

{
  char *infile = "TPV7.fits";

  char keyrec[81], header[288001];
  int  gotend, iblock, ikeyrec, inc, j, k, n, naxis[4], naxis1, naxis2,
       nClosure, nFail, nkeyrec, nreject, nTest, nwcs, p1, p2, status;
  double dp1, dp2, *img, px, *px0, *px1, rel, resid, absmax, relmax;
  FILE *fptr;
  struct linprm *lin;
  struct wcsprm *wcs;


  wcserr_enable(1);
  wcsprintf_set(stdout);

  /* Set line buffering in case stdout is redirected to a file, otherwise
   * stdout and stderr messages will be jumbled (stderr is unbuffered). */
  setvbuf(stdout, NULL, _IOLBF, 0);

  wcsprintf("Testing closure of WCSLIB distortion routines (tdis1.c)\n"
            "-------------------------------------------------------\n");

  /* List status return messages. */
  wcsprintf("\nList of dis status return values:\n");
  for (status = 1; status <= 5; status++) {
    wcsprintf("%4d: %s.\n", status, dis_errmsg[status]);
  }
  wcsprintf("\n");

  /* Optional file name specified? */
  if (1 < argc) {
    infile = argv[1];
  }


  /* Read in the FITS header, excluding COMMENT and HISTORY keyrecords. */
  if ((fptr = fopen(infile, "r")) == 0) {
    wcsprintf("ERROR opening %s\n", infile);
    return 1;
  }

  memset(naxis, 0, 4*sizeof(int));

  k = 0;
  nkeyrec = 0;
  gotend = 0;
  for (iblock = 0; iblock < 100; iblock++) {
    for (ikeyrec = 0; ikeyrec < 36; ikeyrec++) {
      if (fgets(keyrec, 81, fptr) == 0) {
        break;
      }

      if (strncmp(keyrec, "        ", 8) == 0) continue;
      if (strncmp(keyrec, "COMMENT ", 8) == 0) continue;
      if (strncmp(keyrec, "HISTORY ", 8) == 0) continue;

      if (strncmp(keyrec, "NAXIS", 5) == 0) {
        if (keyrec[5] == ' ') continue;
        sscanf(keyrec+5, "%d = %d", &j, &n);
        if (j <= 4) naxis[j-1] = n;
        continue;
      }

      strncpy(header+k, keyrec, 80);
      k += 80;
      nkeyrec++;

      if (strncmp(keyrec, "END       ", 10) == 0) {
        /* An END keyrecord was read, but read the rest of the block. */
        gotend = 1;
      }
    }

    if (gotend) break;
  }
  fclose(fptr);


  /* Parse the header. */
  if ((wcspih(header, nkeyrec, WCSHDR_none, 2, &nreject, &nwcs, &wcs))) {
    wcsperr(wcs, 0x0);
    return 1;
  }


  /* Translate the TPV "projection" into a sequent distortion. */
  if (wcsset(wcs)) {
    wcsperr(wcs, 0x0);
    return 1;
  }

  /* Henceforth, we will work with linprm. */
  lin = &(wcs->lin);


  /* The image size determines the test domain. */
  if ((naxis1 = naxis[wcs->lng]) == 0) {
    naxis1 = 2*wcs->crpix[wcs->lng] + 1;
  }
  if ((naxis2 = naxis[wcs->lat]) == 0) {
    naxis2 = 2*wcs->crpix[wcs->lat] + 1;
  }

  /* Limit the number of tests. */
  inc = 1;
  while ((naxis1/inc)*(naxis2/inc) > 800000) {
    inc *= 2;
  }

  n   = naxis1 / inc;
  px0 = calloc(3*(2*n ), sizeof(double));
  px1 = px0 + 2*n ;
  img = px1 + 2*n ;

  nTest = 0;
  nFail = 0;
  nClosure = 0;
  absmax = 0.0;
  relmax = 0.0;
  for (p2 = 1; p2 <= naxis2; p2 += inc) {
    k = 0;
    for (p1 = 1; p1 <= naxis1; p1 += inc) {
      px0[k++] = (double)p1;
      px0[k++] = (double)p2;
    }

    if (linp2x(lin, n, 2, px0, img)) {
      linperr(lin, 0x0);
      nFail = 1;
      break;
    }

    if (linx2p(lin, n, 2, img, px1)) {
      linperr(lin, 0x0);
      nFail = 1;
      break;
    }

    /* Check closure. */
    k = 0;
    for (k = 0; k < 2*n ; k += 2) {
      dp1 = fabs(px1[k]   - px0[k]);
      dp2 = fabs(px1[k+1] - px0[k+1]);

      resid = (dp1 > dp2) ? dp1 : dp2;
      if (resid > absmax) absmax = resid;

      if (resid > ATOL) {
        nClosure++;
        wcsprintf("   Absolute closure error:\n");
        wcsprintf("    pix: %18.12f %18.12f\n", px0[k], px0[k+1]);
        wcsprintf(" -> img: %18.12f %18.12f\n", img[k], img[k+1]);
        wcsprintf(" -> pix: %18.12f %18.12f\n", px1[k], px1[k+1]);
        wcsprintf("\n");
        continue;
      }

      resid = 0.0;
      if ((px = fabs(px0[k]))   > 1.0) resid = dp1/px;
      if ((px = fabs(px0[k+1])) > 1.0) {
        if ((rel = dp2/px) > resid) resid = rel;
      }
      if (resid > relmax) relmax = resid;

      if (resid > FTOL) {
        nClosure++;
        wcsprintf("   Relative closure error:\n");
        wcsprintf("    pix: %18.12f %18.12f\n", px0[k], px0[k+1]);
        wcsprintf(" -> img: %18.12f %18.12f\n", img[k], img[k+1]);
        wcsprintf(" -> pix: %18.12f %18.12f\n", px1[k], px1[k+1]);
        wcsprintf("\n");
      }
    }

    nTest += n;
  }

  if (nFail) {
    wcsprintf("\nFAIL: The test failed to complete.\n");

  } else {
    wcsprintf("linp2x/linx2p with distortions:\n"
      "  Completed %d closure tests.\n"
      "  Maximum absolute closure residual = %.1e pixel.\n"
      "  Maximum relative closure residual = %.1e.\n", nTest, absmax, relmax);
    wcsprintf("\n");

    if (nClosure) {
      wcsprintf("FAIL: %d closure residuals exceed reporting tolerance.\n",
        nClosure);

    } else {
      wcsprintf("PASS: All closure residuals are within reporting "
        "tolerance.\n");
    }
  }


  free(px0);
  wcsvfree(&nwcs, &wcs);

  return nFail || nClosure;
}
