/*============================================================================

  WCSLIB 5.0 - an implementation of the FITS WCS standard.
  Copyright (C) 1995-2015, Mark Calabretta

  This file is part of WCSLIB.

  WCSLIB is free software: you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free
  Software Foundation, either version 3 of the License, or (at your option)
  any later version.

  WCSLIB is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
  FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
  more details.

  You should have received a copy of the GNU Lesser General Public License
  along with WCSLIB.  If not, see http://www.gnu.org/licenses.

  Direct correspondence concerning WCSLIB to mark@calabretta.id.au

  Author: Mark Calabretta, Australia Telescope National Facility, CSIRO.
  http://www.atnf.csiro.au/people/Mark.Calabretta
  $Id: dis.h,v 5.0 2015/04/05 12:24:59 mcalabre Exp $
*=============================================================================
*
* WCSLIB 5.0 - experimental C routines that implement proposed extensions to
* the FITS World Coordinate System (WCS) standard.  Refer to
*
*   "Representations of distortions in FITS world coordinate systems",
*   Calabretta, M.R. et al. (Paper IV), draft dated 2004/04/22 available from
*   http://www.atnf.csiro.au/people/Mark.Calabretta
*
* Refer to the README file provided with WCSLIB for an overview of the
* library.
*
*
* Summary of the dis routines
* ---------------------------
* These routines apply the distortion functions defined by the extension to
* the FITS WCS standard proposed in Paper IV.  They are based on the disprm
* struct which contains all information needed for the computations.  The
* struct contains some members that must be set by the user, and others that
* are maintained by these routines, somewhat like a C++ class but with no
* encapsulation.
*
* Four routines, disini(), disparms(), discpy(), and disfree() are provided to
* manage the disprm struct, and another, disprt(), prints its contents.
*
* A setup routine, disset(), computes intermediate values in the disprm struct
* from parameters in it that were supplied by the user.  The struct always
* needs to be set up by disset(), though disset() need not be called
* explicitly - refer to the explanation of disprm::flag.
*
* disp2x() and disx2p() implement the WCS distortion functions, disp2x() using
* separate functions, such as tpv1(), to do the computation.
*
*
* disini() - Default constructor for the disprm struct
* ----------------------------------------------------
* disini() allocates memory for arrays in a disprm struct and sets all members
* of the struct to default values.
*
* In general, different distortion functions may be associated with each axis,
* and at the point where disini() is invoked, it is usually not known how many
* distortion parameters may be required for each.  Hence a separate routine,
* disparms(), is provided to allocate memory for the parameter arrays if
* desired.  One way or the other, after invoking disini(), it is the caller's
* responsibility to allocate memory (if necessary) for the distortion
* parameters.  If disparms() is used for this purpose, then disfree() will
* free it and subsequent calls to disini() will manage it, e.g. if naxis
* changes.  Of course, if there is no distortion function on an axis then
* there is no need to allocate memory for any parameters.
*
* PLEASE NOTE: every disprm struct must be initialized by disini(), possibly
* repeatedly.  On the first invokation, and only the first invokation,
* disprm::flag must be set to -1 to initialize memory management, regardless
* of whether disini() will actually be used to allocate memory.
*
* Given:
*   alloc     int       If true, allocate memory unconditionally for arrays in
*                       the disprm struct.
*
*                       If false, it is assumed that pointers to these arrays
*                       have been set by the user except if they are null
*                       pointers in which case memory will be allocated for
*                       them regardless.  (In other words, setting alloc true
*                       saves having to initalize these pointers to zero.)
*
*   naxis     int       The number of world coordinate axes, used to determine
*                       array sizes.
*
* Given and returned:
*   dis       struct disprm*
*                       Distortion function parameters.  Note that, in order
*                       to initialize memory management disprm::flag must be
*                       set to -1 when dis is initialized for the first time
*                       (memory leaks may result if it had already been
*                       initialized).
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* disparms() - Allocate memory for parameters in a disprm struct
* --------------------------------------------------------------
* disparms() allocates memory for the arrays of distortion function parameters
* in a disprm struct.  It relies on disprm::nparm[] having been set to
* determine the number of parameters for each distortion function.  The
* allocated memory will be freed by disfree().
*
* Given and returned:
*   dis       struct disprm*
*                       disprm::nparm[] is used to determine the number of
*                       parameters in each distortion function.  Memory is
*                       allocated for the distortion function parameters,
*                       disprm::parms[][], and it is initialized to zero.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* discpy() - Copy routine for the disprm struct
* ---------------------------------------------
* discpy() does a deep copy of one disprm struct to another, using disini() to
* allocate memory unconditionally for its arrays if required.  Only the
* "information to be provided" part of the struct is copied; a call to
* disset() is required to initialize the remainder.
*
* Given:
*   alloc     int       If true, allocate memory unconditionally for arrays in
*                       the destination, including the parameter array via a
*                       call to disparms().  Otherwise, it is assumed that
*                       pointers to these arrays have been set by the user
*                       except if they are null pointers in which case memory
*                       will be allocated for them regardless.
*
*   dissrc    const struct disprm*
*                       Struct to copy from.
*
* Given and returned:
*   disdst    struct disprm*
*                       Struct to copy to.  disprm::flag should be set to -1
*                       if disdst was not previously initialized (memory leaks
*                       may result if it was previously initialized).
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* disfree() - Destructor for the disprm struct
* --------------------------------------------
* disfree() frees memory allocated for the disprm arrays by disini() and/or
* disparms().  disini() and disparms() keep a record of the memory they
* allocate and disfree() will only attempt to free this.
*
* PLEASE NOTE: disfree() must not be invoked on a disprm struct that was not
* initialized by disini().
*
* Given:
*   dis       struct disprm*
*                       Distortion function parameters.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*
*
* disprt() - Print routine for the disprm struct
* ----------------------------------------------
* disprt() prints the contents of a disprm struct using wcsprintf().  Mainly
* intended for diagnostic purposes.
*
* Given:
*   dis       const struct disprm*
*                       Distortion function parameters.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*
*
* disset() - Setup routine for the disprm struct
* ----------------------------------------------
* disset(), sets up the disprm struct according to information supplied within
* it - refer to the explanation of disprm::flag.
*
* Note that this routine need not be called directly; it will be invoked by
* disp2x() and disx2p() if the disprm::flag is anything other than a
* predefined magic value.
*
* Given and returned:
*   dis       struct disprm*
*                       Distortion function parameters.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*                         3: Invalid parameter.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* disp2x() - Apply distortion function
* ------------------------------------
* disp2x() applies the distortion functions.  By definition, the distortion
* is in the pixel-to-world direction.
*
* Depending on the point in the algorithm chain at which it is invoked,
* disp2x() may transform pixel coordinates to corrected pixel coordinates, or
* intermediate pixel coordinates to corrected intermediate pixel coordinates,
* or image coordinates to corrected image coordinates.
*
*
* Given and returned:
*   dis       struct disprm*
*                       Distortion function parameters.
*
* Given:
*   rawcrd    const double[]
*                       Array of coordinates.
*
* Returned:
*   discrd    double[]  Array of coordinates to which the distortion functions
*                       have been applied.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*                         3: Invalid parameter.
*                         4: Distort error.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* disx2p() - Apply de-distortion function
* ---------------------------------------
* disx2p() applies the inverse of the distortion functions.  By definition,
* the de-distortion is in the world-to-pixel direction.
*
* Depending on the point in the algorithm chain at which it is invoked,
* disx2p() may transform corrected pixel coordinates to pixel coordinates, or
* corrected intermediate pixel coordinates to intermediate pixel coordinates,
* or corrected image coordinates to image coordinates.
*
* disx2p() iteratively solves for the inverse using disp2x().  It assumes
* that the distortion is small and the functions are well-behaved, being
* continuous and with continuous derivatives.  Also that, to first order
* in the neighbourhood of the solution, discrd[j] ~= a + b*rawcrd[j], i.e.
* independent of rawcrd[i], where i != j.  This is effectively equivalent to
* assuming that the distortion functions are separable to first order.
* Furthermore, a is assumed to be small, and b close to unity.
*
* Given and returned:
*   dis       struct disprm*
*                       Distortion function parameters.
*
* Given:
*   discrd    const double[]
*                       Array of coordinates.
*
* Returned:
*   rawcrd    double[]  Array of coordinates to which the inverse distortion
*                       functions have been applied.
*
* Function return value:
*             int       Status return value:
*                         0: Success.
*                         1: Null disprm pointer passed.
*                         2: Memory allocation failed.
*                         3: Invalid parameter.
*                         5: De-distort error.
*
*                       For returns > 1, a detailed error message is set in
*                       disprm::err if enabled, see wcserr_enable().
*
*
* disprm struct - Distortion parameters
* -------------------------------------
* The disprm struct contains all of the information required to apply a set of
* distortion functions.  It consists of certain members that must be set by
* the user ("given") and others that are set by the WCSLIB routines
* ("returned").
*
*   int flag
*     (Given and returned) This flag must be set to zero whenever any of the
*     following members of the disprm struct are set or modified:
*
*       - disprm::naxis,
*       - disprm::dtype,
*       - disprm::axmap,
*       - disprm::nparm,
*       - disprm::parms.
*
*     This signals the initialization routine, disset(), to recompute the
*     returned members of the disprm struct.  disset() will reset flag to
*     indicate that this has been done.
*
*     PLEASE NOTE: flag must be set to -1 when disini() is called for the
*     first time for a particular disprm struct in order to initialize memory
*     management.  It must ONLY be used on the first initialization otherwise
*     memory leaks may result.
*
*   int naxis
*     (Given or returned) Number of pixel and world coordinate elements.
*
*     If disini() is used to initialize the disprm struct (as would normally
*     be the case) then it will set naxis from the value passed to it as a
*     function argument.  The user should not subsequently modify it.
*
*   char (*dtype)[16]
*     (Given) Pointer to the first element of an array of char[16] containing
*     the name of the distortion function for each axis.
*
*   int **axmap
*     (Given) Pointer to the first element of an array of int* containing
*     pointers to the first elements of the axis mapping arrays for each axis.
*
*     An axis mapping associates the independent variables of a distortion
*     function with the 1-relative image axis number.  For example, consider
*     an image with a spectrum on the first axis, followed by RA, Dec, and
*     time axes.  For a distortion in (RA,Dec) and no distortion on the
*     spectral or time axes, the axis mapping arrays, axmap[j][], would be
*
=       j=0: [0, 0, 0, 0]   ...no  distortion on spectral axis,
=         1: [2, 3, 0, 0]   ...RA  distortion depends on RA and Dec,
=         2: [3, 2, 0, 0]   ...Dec distortion depends on Dec and RA,
=         3: [0, 0, 0, 0]   ...no  distortion on time axis,
*
*     where zero indicates that there is no corresponding independent
*     variable.
*
*   double **offset
*     (Given) Pointer to the first element of an array of double* containing
*     an offset used to renormalize the independent variables of the
*     distortion function for each axis.
*
*     The offsets are subtracted from the independent variables before
*     scaling.
*
*     It is not necessary to reset the disprm struct (via disset()) when
*     disprm::offset is changed.
*
*   double **scale
*     (Given) Pointer to the first element of an array of double* containing
*     a scale used to renormalize the independent variables of the distortion
*     function for each axis.
*
*     The scale is applied to the independent variables after the offsets are
*     subtracted.
*
*     It is not necessary to reset the disprm struct (via disset()) when
*     disprm::scale is changed.
*
*   int *nparm
*     (Given) Pointer to the first element of an array of int containing the
*     number of distortion parameters for each axis.
*
*     The number of parameters specified must satisfy the requirements of the
*     particular distortion function.  For example, the polynomial distortion
*     associated with the TPV "projection" must have a number that corresponds
*     to the degree of the polynomial:
*
=       Degree 1:   4
=              2:   7
=              3:  12
=              4:  17
=              5:  24
=              6:  31
=              7:  40
*
*   double **parms
*     (Given) Pointer to the first element of an array of double* containing
*     pointers to the first elements of the arrays of distortion parameters
*     for each axis.
*
*     disparms() may be used to allocate memory for these arrays.
*
*   double *maxdis
*     (Given) Pointer to the first element of an array of double specifying
*     the maximum absolute value of the distortion for each axis computed over
*     the whole image.
*
*     It is not necessary to reset the disprm struct (via disset()) when
*     disprm::maxdis is changed.
*
*   double totdis
*     (Given) The maximum absolute value of the combination of all distortion
*     functions specified as an offset in pixel coordinates computed over the
*     whole image.
*
*     It is not necessary to reset the disprm struct (via disset()) when
*     disprm::totdis is changed.
*
*   struct wcserr *err
*     (Returned) If enabled, when an error status is returned, this struct
*     contains detailed information about the error, see wcserr_enable().
*
*   int (**disp2x)(DISP2X_ARGS)
*     (For internal use only.)
*   int (**disx2p)(DISX2P_ARGS)
*     (For internal use only.)
*   double *tmpmem
*     (For internal use only.)
*   int **iwrk
*     (For internal use only.)
*   double **dwrk
*     (For internal use only.)
*   int m_flag
*     (For internal use only.)
*   int m_naxis
*     (For internal use only.)
*   char (*m_dtype)[16]
*     (For internal use only.)
*   int **m_axmap
*     (For internal use only.)
*   double **m_offset
*     (For internal use only.)
*   double **m_scale
*     (For internal use only.)
*   int *m_nparm
*     (For internal use only.)
*   double **m_parms
*     (For internal use only.)
*   double *m_maxdis
*     (For internal use only.)
*   int m_alloc
*     (For internal use only.)
*   int m_padding
*     (For internal use only.)
*
*
* Global variable: const char *dis_errmsg[] - Status return messages
* ------------------------------------------------------------------
* Error messages to match the status value returned from each function.
*
*===========================================================================*/

#ifndef WCSLIB_DIS
#define WCSLIB_DIS

#ifdef __cplusplus
extern "C" {
#endif


extern const char *dis_errmsg[];

enum dis_errmsg_enum {
  DISERR_SUCCESS      = 0,	/* Success. */
  DISERR_NULL_POINTER = 1,	/* Null disprm pointer passed. */
  DISERR_MEMORY       = 2,	/* Memory allocation failed. */
  DISERR_BAD_PARAM    = 3,	/* Invalid parameter value. */
  DISERR_DISTORT      = 4,	/* Distortion error. */
  DISERR_DEDISTORT    = 5	/* De-distortion error. */
};

/* For use in declaring distortion function prototypes. */
#define DISP2X_ARGS int nparm, const double parms[], int ncrd, \
const double rawcrd[], double *discrd

/* For use in declaring de-distortion function prototypes. */
#define DISX2P_ARGS int nparm, const double parms[], int ncrd, \
const double discrd[], double rawcrd[]


struct disprm {
  /* Initialization flag (see the prologue above).                          */
  /*------------------------------------------------------------------------*/
  int flag;			/* Set to zero to force initialization.     */

  /* Parameters to be provided (see the prologue above).                    */
  /*------------------------------------------------------------------------*/
  int naxis;			/* The number of pixel coordinate elements, */
				/* given by NAXIS.                          */
  char   (*dtype)[16];		/* For each axis, the distortion type.      */
  int    **axmap;		/* For each axis, the axis mapping array.   */
  double **offset;		/* For each axis, renormalization offset.   */
  double **scale;		/* For each axis, renormalization scale.    */
  int    *nparm;		/* For each axis, the number of distortion  */
				/* parameters.                              */
  double **parms;		/* For each axis, the array of distortion   */
				/* parameters.                              */
  double *maxdis;		/* For each axis, the maximum distortion.   */
  double totdis;		/* The maximum combined distortion.         */

  /* Error handling, if enabled.                                            */
  /*------------------------------------------------------------------------*/
  struct wcserr *err;

  /* Private - the remainder are for internal use.                          */
  /*------------------------------------------------------------------------*/
  int (**disp2x)(DISP2X_ARGS);	/* For each axis, pointers to the           */
  int (**disx2p)(DISX2P_ARGS);	/* distortion function and its inverse.     */

  double *tmpmem;
  double **iwrk;
  double **dwrk;

  int    m_flag, m_naxis;	/* The remainder are for memory management. */
  char   (*m_dtype)[16];
  int    **m_axmap;
  double **m_offset, **m_scale;
  int    *m_nparm;
  double **m_parms;
  double *m_maxdis;
  int    m_alloc, m_padding;
};

/* Size of the disprm struct in int units, used by the Fortran wrappers. */
#define DISLEN (sizeof(struct disprm)/sizeof(int))


int disini(int alloc, int naxis, struct disprm *dis);

int disparms(struct disprm *dis);

int discpy(int alloc, const struct disprm *dissrc, struct disprm *disdst);

int disfree(struct disprm *dis);

int disprt(const struct disprm *dis);

int disset(struct disprm *dis);

int disp2x(struct disprm *dis, const double rawcrd[], double discrd[]);

int disx2p(struct disprm *dis, const double discrd[], double rawcrd[]);


/* Specialist distortion functions (internal use only). */
int tpvset(int j, struct disprm *dis);
int tpv1(DISP2X_ARGS);
int tpv2(DISP2X_ARGS);
int tpv3(DISP2X_ARGS);
int tpv4(DISP2X_ARGS);
int tpv5(DISP2X_ARGS);
int tpv6(DISP2X_ARGS);
int tpv7(DISP2X_ARGS);

#ifdef __cplusplus
}
#endif

#endif /* WCSLIB_DIS */
